package ru.yandex.solomon.math.stat;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.solomon.model.point.column.HistogramColumn;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class HistogramAggr {

    private static class Aver {
        private double sum;
        private double count;

        public static final Aver EMPTY = new Aver();

        Aver(double sum, double count) {
            this.sum = sum;
            this.count = count;
        }

        Aver() {
            this(0d, 0d);
        }

        private void add(double count, double mean) {
            this.count += count;
            this.sum += count * mean;
        }

        double getSum() {
            return sum;
        }

        double getAvg() {
            return sum / count;
        }

        double getCount() {
            return count;
        }
    }

    private static Aver collect(HistogramLike histogram) {
        Aver result = new Aver();
        double lowerLimit = 0;
        double upperLimit;
        int i = 0;
        for (; i < histogram.count() - 1; i++) {
            upperLimit = histogram.upperBound(i);
            result.add(histogram.value(i), (upperLimit + lowerLimit) * 0.5);
            lowerLimit = upperLimit;
        }
        upperLimit = histogram.upperBound(i);
        if (upperLimit == Histograms.INF_BOUND) {
            result.add(histogram.value(i), lowerLimit);
        } else {
            result.add(histogram.value(i), (upperLimit + lowerLimit) * 0.5);
        }
        return result;
    }

    /*
     * NATIVE HISTOGRAMS
     */

    public static double sum(@Nullable Histogram histogram) {
        if (HistogramColumn.isNullOrEmpty(histogram)) {
            return Aver.EMPTY.getSum();
        }

        return collect(HistogramLikeFactory.ofHist(histogram)).getSum();
    }

    public static double avg(@Nullable Histogram histogram) {
        if (HistogramColumn.isNullOrEmpty(histogram)) {
            return Aver.EMPTY.getAvg();
        }

        return collect(HistogramLikeFactory.ofHist(histogram)).getAvg();
    }

    /*
     * LOG HISTOGRAMS
     */

    private static Aver collect(@Nullable LogHistogram logHistogram) {
        Aver result = new Aver();
        if (logHistogram == null) {
            return result;
        }

        result.add(logHistogram.getCountZero(), 0);

        final double base = logHistogram.getBase();
        double bucketMean = Math.pow(base, logHistogram.getStartPower() + 0.5);
        for (int i = 0; i < logHistogram.countBucket(); i++) {
            result.add(logHistogram.getBucketValue(i), bucketMean);
            bucketMean *= base;
        }
        return result;
    }

    public static double sum(@Nullable LogHistogram histogram) {
        return collect(histogram).getSum();
    }

    public static double avg(@Nullable LogHistogram histogram) {
        return collect(histogram).getAvg();
    }

    /*
     * LEGACY HISTOGRAMS
     */

    public static double sum(LegacyHistogramPoint point) {
        return collect(HistogramLikeFactory.ofLegacy(point)).getSum();
    }

    public static double avg(LegacyHistogramPoint point) {
        return collect(HistogramLikeFactory.ofLegacy(point)).getAvg();
    }
}
