package ru.yandex.solomon.math.stat;

import java.util.Arrays;

import javax.annotation.Nullable;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Ivan Tsybulin
 */
public final class HistogramCumulativeDistribution {

    private HistogramCumulativeDistribution() {
    }

    static void cumulativeCount(
            HistogramLike histogram,
            double[] cumulativeCount,
            double[] sortedBounds)
    {
        if (sortedBounds.length == 0) {
            return;
        }

        if (histogram.count() == 0) {
            Arrays.fill(cumulativeCount, 0);
            return;
        }

        double prevBucketBound = 0;
        int boundIdx = 0;
        double countInPrevBuckets = 0;

        while (sortedBounds[boundIdx] <= prevBucketBound) {
            cumulativeCount[boundIdx] = countInPrevBuckets;
            boundIdx++;
            if (boundIdx == sortedBounds.length) {
                return;
            }
        }

        for (int i = 0; i < histogram.count(); i++) {
            double currBucketBound = histogram.upperBound(i);
            double currBucketCount = histogram.value(i);

            while (sortedBounds[boundIdx] < currBucketBound) {
                cumulativeCount[boundIdx] = interpolateCount(
                        prevBucketBound,
                        currBucketBound,
                        sortedBounds[boundIdx],
                        countInPrevBuckets,
                        currBucketCount);
                boundIdx++;

                if (boundIdx == sortedBounds.length) {
                    return;
                }
            }

            prevBucketBound = currBucketBound;
            countInPrevBuckets += currBucketCount;
        }

        for (; boundIdx < sortedBounds.length; boundIdx++) {
            cumulativeCount[boundIdx] = countInPrevBuckets;
        }
    }

    private static double interpolateCount(
            double prevBucketBound,
            double currBucketBound,
            double sortedBound,
            double countInPrevBuckets,
            double currBucketCount)
    {
        if (currBucketBound == Histograms.INF_BOUND) {
            return countInPrevBuckets; // no good choice here
        }
        double bucketFraction = (sortedBound - prevBucketBound) / (currBucketBound - prevBucketBound);
        return countInPrevBuckets + bucketFraction * currBucketCount;
    }

    public static void cumulativeCount(LegacyHistogramPoint histogram, double[] cumulativeCount, double[] sortedBounds) {
        cumulativeCount(HistogramLikeFactory.ofLegacy(histogram), cumulativeCount, sortedBounds);
    }

    @VisibleForTesting
    public static double[] cumulativeCount(@Nullable Histogram histogram, double[] sortedBounds) {
        double[] cumulativeCount = new double[sortedBounds.length];
        cumulativeCount(histogram, cumulativeCount, sortedBounds);
        return cumulativeCount;
    }

    public static void cumulativeCount(@Nullable Histogram histogram, double[] cumulativeCount, double[] sortedBounds) {
        if (histogram == null) {
            Arrays.fill(cumulativeCount, 0d);
            return;
        }

        cumulativeCount(HistogramLikeFactory.ofHist(histogram), cumulativeCount, sortedBounds);
    }

    public static void cumulativeCount(@Nullable LogHistogram histogram, double[] cumulativeCount, double[] sortedBounds) {
        if (histogram == null) {
            Arrays.fill(cumulativeCount, 0d);
            return;
        }

        int boundIdx = 0;

        while (sortedBounds[boundIdx] <= 0) {
            cumulativeCount[boundIdx] = 0;
            boundIdx++;
            if (boundIdx == sortedBounds.length) {
                return;
            }
        }

        double countInPrevBuckets = histogram.getCountZero();
        double base = histogram.getBase();
        int startPower = histogram.getStartPower();
        double bucketLowerLimit = Math.pow(base, startPower);
        double bucketUpperLimit = bucketLowerLimit * base;
        double invLogBase = 1 / Math.log(base);

        while (sortedBounds[boundIdx] <= bucketLowerLimit) {
            cumulativeCount[boundIdx] = countInPrevBuckets;
            boundIdx++;
            if (boundIdx == sortedBounds.length) {
                return;
            }
        }

        for (int i = 0; i < histogram.countBucket(); i++) {
            double currBucketCount = histogram.getBucketValue(i);
            if (Double.isNaN(currBucketCount)) {
                currBucketCount = 0;
            }

            while (sortedBounds[boundIdx] < bucketUpperLimit) {
                double bucketFraction = Math.log(sortedBounds[boundIdx] / bucketLowerLimit) * invLogBase;
                cumulativeCount[boundIdx] = countInPrevBuckets + bucketFraction * currBucketCount;

                boundIdx++;

                if (boundIdx == sortedBounds.length) {
                    return;
                }
            }

            countInPrevBuckets += currBucketCount;
            bucketLowerLimit = bucketUpperLimit;
            bucketUpperLimit *= base;
        }

        for (; boundIdx < sortedBounds.length; boundIdx++) {
            cumulativeCount[boundIdx] = countInPrevBuckets;
        }
    }

    @VisibleForTesting
    public static double[] cumulativeCount(@Nullable LogHistogram histogram, double[] sortedBounds) {
        double[] cumulativeCount = new double[sortedBounds.length];
        cumulativeCount(histogram, cumulativeCount, sortedBounds);
        return cumulativeCount;
    }
}
