package ru.yandex.solomon.math.stat;


import java.util.Arrays;

import javax.annotation.Nullable;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.solomon.model.point.column.HistogramColumn;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.util.collection.array.DoubleArrayView;


/**
 * @author Vladimir Gordiychuk
 */
public final class HistogramPercentile {
    static final double EMPTY_HISTOGRAM_PERCENTILE_VALUE = 0;

    private HistogramPercentile() {
    }

    static void percentilesAtTimePoint(double[] percents, double[] buckets, double[] bounds, double[] result) {
        double sum = 0;
        boolean hasData = false;
        for (double value : buckets) {
            if (!Double.isNaN(value)) {
                hasData = true;
                sum += value;
            }
        }

        if (!hasData) {
            Arrays.fill(result, Double.NaN);
            return;
        }

        for (int i = 0; i < percents.length; ++i) {
            double percent = percents[i];
            result[i] = percentile(new DoubleArrayView(buckets), bounds, sum, percent);
        }
    }

    public static void percentiles(LegacyHistogramPoint point, double[] percentiles, double[] percentileLevels) {
        if (point.bucketLimits.length == 0) {
            Arrays.fill(percentiles, HistogramPercentile.EMPTY_HISTOGRAM_PERCENTILE_VALUE);
            return;
        }

        double sum = 0;
        for (double value : point.bucketValues) {
            sum += value;
        }

        var bucketsView = new DoubleArrayView(point.bucketValues);
        for (int i = 0; i < percentiles.length; i++) {
            percentiles[i] = HistogramPercentile.percentile(bucketsView, point.bucketLimits, sum, percentileLevels[i]);
        }
    }

    @VisibleForTesting
    public static double[] percentiles(@Nullable Histogram histogram, double[] percentileLevels) {
        double[] percentiles = new double[percentileLevels.length];
        percentiles(histogram, new double[Histograms.MAX_BUCKETS_COUNT], percentiles, percentileLevels);
        return percentiles;
    }

    public static void percentiles(@Nullable Histogram histogram, double[] buckets, double[] percentiles, double[] percentileLevels) {
        if (HistogramColumn.isNullOrEmpty(histogram)) {
            Arrays.fill(percentiles, HistogramPercentile.EMPTY_HISTOGRAM_PERCENTILE_VALUE);
            return;
        }

        double sum = 0;
        for (int i = 0; i < histogram.count(); i++) {
            buckets[i] = histogram.valueDivided(i);
            sum += buckets[i];
        }

        var bucketsView = new DoubleArrayView(buckets, 0, histogram.count());
        double[] bounds = histogram.getBounds();
        for (int i = 0; i < percentiles.length; i++) {
            percentiles[i] = HistogramPercentile.percentile(bucketsView, bounds, sum, percentileLevels[i]);
        }
    }

    private static double percentile(DoubleArrayView buckets, double[] bounds, double sum, double percentile) {
        double amountCutoffForPercentile = getAmountCutoffForPercentile(sum, percentile);
        double currentBucketBound = 0;
        double totalToCurrentIndex = 0;
        boolean first = true;

        if (Double.compare(sum, 0) <= 0) {
            return EMPTY_HISTOGRAM_PERCENTILE_VALUE;
        }

        for (int index = 0; index < buckets.length(); index++) {
            double currentCountAtBucket = buckets.at(index);
            if (Double.isNaN(currentCountAtBucket)) {
                continue;
            }

            double prevBucketBound = currentBucketBound;
            currentBucketBound = bounds[index];

            double totalToPrevIndex = totalToCurrentIndex;
            totalToCurrentIndex += currentCountAtBucket;
            if (totalToCurrentIndex > 0) {
                first = false;
            }

            int compare = Double.compare(totalToCurrentIndex, amountCutoffForPercentile);
            if ((first && compare > 0) || (!first && compare >= 0)) {
                return computeInterpolatedBucketBound(
                    amountCutoffForPercentile,
                    currentBucketBound,
                    totalToCurrentIndex,
                    prevBucketBound,
                    totalToPrevIndex);
            }
        }

        return currentBucketBound;
    }

    private static double getAmountCutoffForPercentile(double sum, double percentile) {
        if (Double.compare(percentile, 100d) >= 0) {
            return sum;
        } else {
            return Math.min(sum * percentile / 100d, sum);
        }
    }

    private static double computeInterpolatedBucketBound(
            double countAtPercentile,
            double currentBucketBound,
            double totalToCurrentIndex,
            double prevBucketBound,
            double totalToPrevIndex)
    {
        if (Double.compare(currentBucketBound, Histograms.INF_BOUND) == 0) {
            if (totalToCurrentIndex == 0d) {
                return EMPTY_HISTOGRAM_PERCENTILE_VALUE;
            }
            return prevBucketBound;
        }

        if (totalToCurrentIndex == totalToPrevIndex) {
            return currentBucketBound;
        }

        double part = (countAtPercentile - totalToPrevIndex) / (totalToCurrentIndex - totalToPrevIndex);
        return prevBucketBound + part * (currentBucketBound - prevBucketBound);
    }

    @VisibleForTesting
    public static double[] percentiles(@Nullable LogHistogram histogram, double[] percentileLevels) {
        double[] percentiles = new double[percentileLevels.length];
        percentiles(histogram, new double[LogHistogram.LIMIT_MAX_BUCKETS_SIZE], percentiles, percentileLevels);
        return percentiles;
    }

    public static void percentiles(@Nullable LogHistogram histogram, double[] buckets, double[] percentiles, double[] percentileLevels) {
        if (histogram == null) {
            Arrays.fill(percentiles, HistogramPercentile.EMPTY_HISTOGRAM_PERCENTILE_VALUE);
            return;
        }

        double sum = histogram.getCountZero();
        int lastNonzero = -1;
        for (int i = 0; i < histogram.countBucket(); i++) {
            buckets[i] = histogram.getBucketValue(i);
            if (buckets[i] > 0) {
                lastNonzero = i;
            }
            sum += buckets[i];
        }

        DoubleArrayView bucketsView = new DoubleArrayView(buckets, 0, lastNonzero + 1);

        for (int i = 0; i < percentiles.length; i++) {
            percentiles[i] = HistogramPercentile.logHistPercentile(
                    histogram.getCountZero(),
                    bucketsView,
                    histogram.getStartPower(),
                    histogram.getBase(),
                    sum,
                    percentileLevels[i]);
        }
    }

    private static double logHistPercentile(
            long countZero,
            DoubleArrayView buckets,
            int startPower,
            double base,
            double total,
            double pecentLevel)
    {
        if (Double.compare(total, 0) <= 0) {
            return EMPTY_HISTOGRAM_PERCENTILE_VALUE;
        }

        final double lookup = getAmountCutoffForPercentile(total, pecentLevel);
        if (lookup < countZero) {
            return 0;
        }

        double bucketLowerLimit = Math.pow(base, startPower);
        double bucketUpperLimit = Math.pow(base, startPower + 1);
        double beforeSum = countZero;
        for (int i = 0; i < buckets.length(); i++) {
            double countAtBucket = buckets.at(i);
            if (Double.isNaN(countAtBucket)) {
                countAtBucket = 0;
            }

            double afterSum = beforeSum + countAtBucket;

            if (countAtBucket > 0) {
                if (lookup <= afterSum) {
                    double bucketFraction = (lookup - beforeSum) / countAtBucket;

                    return bucketLowerLimit * Math.pow(base, bucketFraction);
                }
            }

            beforeSum = afterSum;
            bucketLowerLimit = bucketUpperLimit;
            bucketUpperLimit *= base;
        }

        return bucketLowerLimit;
    }
}
