package ru.yandex.solomon.math.stat;

import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.logging.log4j.util.TriConsumer;

import ru.yandex.solomon.model.point.HasMutableTs;
import ru.yandex.solomon.model.timeseries.iterator.GenericIterator;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public abstract class HistogramPercentileIterator<HistogramPoint extends HasMutableTs> implements GenericIterator<VectorPoint> {
    private final GenericIterator<HistogramPoint> iterator;
    private final double[] percentileLevels;
    private final HistogramPoint point;

    protected HistogramPercentileIterator(GenericIterator<HistogramPoint> iterator, double[] percentileLevels) {
        this.iterator = iterator;
        this.percentileLevels = percentileLevels;
        this.point = newPoint();
    }

    public static <H extends HasMutableTs> HistogramPercentileIterator<H> of(
            GenericIterator<H> iterator,
            double[] percentileLevels,
            Supplier<H> factory,
            TriConsumer<H, double[], double[]> computePercentiles)
    {
        return new HistogramPercentileIterator<>(iterator, percentileLevels) {
            @Override
            public H newPoint() {
                return factory.get();
            }

            @Override
            public void computePercentiles(H histogram, double[] percentiles, double[] percentileLevels) {
                computePercentiles.accept(histogram, percentiles, percentileLevels);
            }
        };
    }

    @Override
    public boolean next(VectorPoint target) {
        if (!iterator.next(point)) {
            return false;
        }

        target.tsMillis = point.getTsMillis();
        computePercentiles(point, target.values, percentileLevels);

        return true;
    }

    public abstract HistogramPoint newPoint();

    public abstract void computePercentiles(HistogramPoint point, double[] percentiles, double[] percentileLevels);

    @Override
    public int estimatePointsCount() {
        return iterator.estimatePointsCount();
    }
}
