package ru.yandex.solomon.math.stat;

import java.util.function.Supplier;
import java.util.function.ToDoubleFunction;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.point.HasMutableTs;
import ru.yandex.solomon.model.point.MutableDataPoint;
import ru.yandex.solomon.model.timeseries.iterator.GenericIterator;

/**
 * Reduce histogram point to a single scalar value
 *
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public abstract class HistogramReduceIterator<HistogramPoint extends HasMutableTs> implements GenericIterator<MutableDataPoint> {
    private final GenericIterator<HistogramPoint> iterator;
    private final HistogramPoint readPoint;

    private HistogramReduceIterator(GenericIterator<HistogramPoint> iterator) {
        this.iterator = iterator;
        this.readPoint = newPoint();
    }

    public abstract HistogramPoint newPoint();

    public abstract double reduce(HistogramPoint point);

    public static <H extends HasMutableTs> HistogramReduceIterator<H> of(
            GenericIterator<H> iterator,
            Supplier<H> factory,
            ToDoubleFunction<H> reducer)
    {
        return new HistogramReduceIterator<>(iterator) {
            @Override
            public H newPoint() {
                return factory.get();
            }

            @Override
            public double reduce(H histogram) {
                return reducer.applyAsDouble(histogram);
            }
        };
    }

    @Override
    public boolean next(MutableDataPoint output) {
        if (!iterator.next(readPoint)) {
            return false;
        }

        output.setTsMillis(readPoint.getTsMillis());
        output.setValue(reduce(readPoint));

        return true;
    }

    @Override
    public int estimatePointsCount() {
        return iterator.estimatePointsCount();
    }
}
