package ru.yandex.solomon.math.stat;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.timeseries.iterator.GenericCombineIterator;
import ru.yandex.solomon.model.timeseries.iterator.GenericIterator;
import ru.yandex.solomon.model.timeseries.iterator.GenericPointCollector;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class LegacyHistogramIterator implements GenericIterator<LegacyHistogramPoint> {
    private final GenericCombineIterator<HistogramBucketCursor, LegacyHistogramPoint> iterator;

    private static class HistogramToVectorCollector implements GenericPointCollector<HistogramBucketCursor, LegacyHistogramPoint> {
        private final double[] bucketLimits;
        private final double[] bucketValues;
        private final int[] bucketIdxByCursorIdx;

        HistogramToVectorCollector(double[] bucketLimits) {
            // Map buckets with same bucketLimit into single bucket
            bucketIdxByCursorIdx = new int[bucketLimits.length];
            List<Double> mergedBuckets = new ArrayList<>();
            int currentBucketId = -1;
            for (int cursorIdx = 0; cursorIdx < bucketLimits.length; cursorIdx++) {
                if (currentBucketId < 0 || mergedBuckets.get(currentBucketId) != bucketLimits[cursorIdx]) {
                    mergedBuckets.add(bucketLimits[cursorIdx]);
                    currentBucketId++;
                }
                bucketIdxByCursorIdx[cursorIdx] = currentBucketId;
            }
            this.bucketLimits = mergedBuckets.stream().mapToDouble(Double::doubleValue).toArray();
            this.bucketValues = new double[this.bucketLimits.length];
        }

        @Override
        public void reset() {
            Arrays.fill(bucketValues, 0);
        }

        @Override
        public void append(int cursorIndex, HistogramBucketCursor cursor) {
            double value = cursor.getPoint().getValueDivided();
            if (!Double.isNaN(value)) {
                bucketValues[bucketIdxByCursorIdx[cursorIndex]] += value;
            }
        }

        @Override
        public void compute(long timestamp, LegacyHistogramPoint target) {
            target.setTsMillis(timestamp);
            target.copyValues(bucketValues);
            target.bucketLimits = bucketLimits;
        }
    }

    public LegacyHistogramIterator(List<HistogramBucketCursor> cursors) {
        var bucketLimits = cursors.stream().mapToDouble(HistogramBucketCursor::getBucketLimit).toArray();
        this.iterator = new GenericCombineIterator<>(cursors, new HistogramToVectorCollector(bucketLimits));
    }

    @Override
    public boolean next(LegacyHistogramPoint target) {
        return iterator.next(target);
    }

    @Override
    public int estimatePointsCount() {
        return iterator.estimatePointsCount();
    }
}
