package ru.yandex.solomon.math.stat;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.math3.stat.descriptive.rank.Percentile;

import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.GraphDataArrayList;
import ru.yandex.solomon.util.collection.array.DoubleArrayView;
import ru.yandex.solomon.util.collection.array.LongArrayView;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class MovingPercentile {

    /**
     * Simple moving percentile is the unweighted percentile of the previous N data inside interval.
     * @param rank value of percentile, should be in interval from 0 to 100, for example 99.9
     */
    public static GraphData simple(GraphData graphData, Duration window, double rank) {
        if (rank < 0 || rank > 100) {
            throw new IllegalArgumentException("Rang should be in interval from 0 to 100");
        }

        long windowMillis = window.toMillis();
        GraphData noNanGraphData = graphData.filterNonNan();

        LongArrayView times = noNanGraphData.getTimestamps();
        DoubleArrayView valuesView = noNanGraphData.getValues();
        int length = noNanGraphData.length();

        GraphDataArrayList result = new GraphDataArrayList();

        int windowCount = 0;
        int indexWindowStart = 0;

        Percentile percentile = new Percentile();
        for (int index = 0; index < length; index++) {
            long pointTime = times.at(index);
            while (pointTime - times.at(indexWindowStart)  > windowMillis) {
                windowCount--;
                indexWindowStart++;
            }

            windowCount++;

            double target = percentile.evaluate(valuesView.array, indexWindowStart + valuesView.from, windowCount, rank);
            result.add(pointTime, target);
        }

        return result.buildGraphData();
    }
}
