package ru.yandex.solomon.math.stat;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.GraphDataArrayList;
import ru.yandex.solomon.util.collection.array.DoubleArrayView;
import ru.yandex.solomon.util.collection.array.LongArrayView;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class MovingSumOrAvg {
    private MovingSumOrAvg() {
    }

    /**
     * Simple moving sum (SMS) is the sum of the previous N data inside interval.
     */
    public static GraphData simpleSum(GraphData graphData, Duration window) {
        return simple(graphData, window, false);
    }

    public static GraphData simpleAvg(GraphData graphData, Duration window) {
        return simple(graphData, window, true);
    }

    // using https://en.wikipedia.org/wiki/Kahan_summation_algorithm
    // to eliminate loss of positiveness
    private static class KahanBabushkaNeumaierAccumulator {
        private double sum = 0;
        private double c = 0;

        public void add(double v) {
            double t = sum + v;
            if (Math.abs(sum) >= Math.abs(v)) {
                c += (sum - t) + v;
            } else {
                c += (v - t) + sum;
            }
            sum = t;
        }

        public double get() {
            return sum + c;
        }

        @Override
        public String toString() {
            return "KahanBabushkaNeumaierAccumulator[sum=" + sum + ", c=" + c + "]";
        }
    }

    private static GraphData simple(GraphData graphData, Duration window, boolean normalize) {
        long windowMillis = window.toMillis();
        GraphData noNanGraphData = graphData.filterNonNan();

        LongArrayView times = noNanGraphData.getTimestamps();
        DoubleArrayView values = noNanGraphData.getValues();
        int length = noNanGraphData.length();

        GraphDataArrayList result = new GraphDataArrayList();

        KahanBabushkaNeumaierAccumulator windowSum = new KahanBabushkaNeumaierAccumulator();
        int indexWindowStart = 0;
        int windowCount = 0;
        for (int index = 0; index < length; index++) {
            long pointTime = times.at(index);
            while (pointTime - times.at(indexWindowStart)  > windowMillis) {
                windowSum.add(-values.at(indexWindowStart));
                windowCount--;
                indexWindowStart++;

                System.out.println(windowSum);
            }

            windowSum.add(values.at(index));
            windowCount++;

            System.out.println(windowSum);

            result.add(pointTime, normalize ? windowSum.get() / windowCount : windowSum.get());
        }

        return result.buildGraphData();
    }
}

