package ru.yandex.solomon.math.stat;

import java.util.Arrays;

import javax.annotation.ParametersAreNonnullByDefault;

@ParametersAreNonnullByDefault
abstract class OutlierFilter {
    // ArrayList does not allow accesses to its underlying buffer
    private final static int DEFAULT_CAPACITY = 200;
    private int size = 0;
    private double[] data = new double[DEFAULT_CAPACITY];

    private boolean finalized = false;

    private final double dropFraction;

    OutlierFilter(double dropFraction) {
        if (dropFraction < 0 || dropFraction >= 1)
            throw new IllegalArgumentException("dropFraction should be in [0, 1), but was " + dropFraction);
        this.dropFraction = dropFraction;
    }

    public int getSize() {
        return size;
    }

    protected abstract void processFiltered(final double[] data, int size, int leftMargin, int rightMargin);

    final protected void process() {
        if (size == 0) {
            finalized = true;
            return;
        }

        if (finalized) {
            return;
        }

        final int dropCount = (int)(dropFraction * size);
        final int dropLeft = dropCount / 2;
        final int dropRight = dropCount - dropLeft;
        final int leftMargin = dropLeft;
        final int rightMargin = size - dropRight - 1;

        PartialOrderer.reorderOutliers(data, 0, size, leftMargin, rightMargin);

        processFiltered(data, size, leftMargin, rightMargin);

        finalized = true;
    }

    public void add(double v) {
        if (finalized) {
            throw new AssertionError("Filter has already been finalized. No adding is possible");
        }

        int oldCapacity = data.length;
        if (size == oldCapacity) {
            int newCapacity = 1 + oldCapacity + (oldCapacity >> 1);
            data = Arrays.copyOf(data, newCapacity);
        }
        data[size] = v;
        size++;
    }
}
