package ru.yandex.solomon.math.stat.trends;

import java.time.Duration;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.SortedOrCheck;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class TrendLineUtils {

    public static GraphData computeTrendLine(
        GraphData graphData,
        Duration fromDuration,
        Duration toDuration,
        Function<GraphData, TrendLine> trendLineCreator)
    {
        if (fromDuration.compareTo(toDuration) >= 0) {
            throw new IllegalStateException("toDuration must be greater than fromDuration");
        }

        long[] timestamps = graphData.getTimestamps().array;
        double[] values = graphData.getValues().array;

        // it's impossible to predict trend line for source with <= 2 points
        if (values.length <= 2) {
            return GraphData.empty;
        }

        long lastGraphPointMillis = timestamps[timestamps.length - 1];

        TrendLine trendLine = trendLineCreator.apply(graphData);

        if (!trendLine.canPredict()) {
            return GraphData.empty;
        }

        long stepMillis = (toDuration.toMillis() - fromDuration.toMillis()) / 1000;

        if (stepMillis < 1000) {
            stepMillis = 1000;
        }

        long startMillis = lastGraphPointMillis + fromDuration.toMillis();
        long endMillis = lastGraphPointMillis + toDuration.toMillis();

        startMillis = startMillis + (startMillis + stepMillis) % stepMillis;
        endMillis = endMillis - endMillis % stepMillis;

        int count = (int) (Math.abs(endMillis - startMillis) / (stepMillis + 1));

        long[] trendTimestamps = new long[count];
        double[] trendValues = new double[count];

        long curMillis = startMillis;

        for (int i = 0; i < count; ++i) {
            trendTimestamps[i] = curMillis;
            trendValues[i] = trendLine.predict(curMillis);
            curMillis += stepMillis;
        }

        return new GraphData(trendTimestamps, trendValues, SortedOrCheck.SORTED_UNIQUE);
    }
}
