package ru.yandex.solomon.model;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricKey {
    private static final MetricKey UNKNOWN = new MetricKey(
            MetricType.UNKNOWN,
            "",
            Labels.empty(),
            0,
            Collections.emptyList());

    private final String name;
    private final MetricType type;
    private final Labels labels;
    // TODO: drop it when old data api will be completely dropped (gordiychuk@)
    private final long createdAtMillis;
    private final List<StockpileKey> stockpileKeys;

    public MetricKey(
        MetricType type,
        Labels labels,
        StockpileKey stockpileKey)
    {
        this(type, "", labels, 0, List.of(stockpileKey));
    }

    public MetricKey(MetricType type, String name, Labels labels, StockpileKey stockpileKey) {
        this(type, name, labels, 0, List.of(stockpileKey));
    }

    public MetricKey(MetricType type, String name, Labels labels, long createdAtMillis, List<StockpileKey> stockpileKeys) {
        this.type = type;
        this.name = name;
        this.labels = labels;
        this.createdAtMillis = createdAtMillis;
        this.stockpileKeys = stockpileKeys;
    }

    @Nonnull
    public static MetricKey orUnknown(@Nullable MetricKey key) {
        if (key == null) {
            return UNKNOWN;
        }
        return key;
    }

    public static MetricKey unknown() {
        return UNKNOWN;
    }

    public MetricType getType() {
        return type;
    }

    public String getName() {
        return name;
    }

    public Labels getLabels() {
        return labels;
    }

    public List<StockpileKey> getStockpileKeys() {
        return stockpileKeys;
    }

    public long getCreatedAtMillis() {
        return createdAtMillis;
    }

    public MetricKey combine(MetricKey target) {
        if (!name.equals(target.name)) {
            throw new IllegalStateException("Not able combine key with different names: " + name + " != " + target.name);
        }

        if (!labels.equals(target.labels)) {
            throw new IllegalStateException("Not able combine key with different labels: " + labels + " != " + target.labels);
        }

        List<StockpileKey> combineStockpileKeys = Stream.concat(stockpileKeys.stream(), target.stockpileKeys.stream())
                .distinct()
                .collect(Collectors.toList());

        return new MetricKey(type, name, labels, Math.max(createdAtMillis, target.createdAtMillis), combineStockpileKeys);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        MetricKey key = (MetricKey) o;

        if (type != key.type) return false;
        if (!name.equals(key.name)) return false;
        return labels.equals(key.labels);
    }

    @Override
    public int hashCode() {
        int result = type.hashCode();
        result = 31 * result + name.hashCode();
        result = 31 * result + labels.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "SensorKey{" +
                "type=" + type +
                ", name=" + name +
                ", labels=" + labels +
                ", stockpileKeys=" + stockpileKeys +
                '}';
    }
}
