package ru.yandex.solomon.model.array.mh;

import java.lang.invoke.MethodHandle;
import java.util.Arrays;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.type.array.ArrayType;
import ru.yandex.commune.mh.builder.MhCall;
import ru.yandex.commune.mh.builder.MhConst;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.commune.mh.builder.MhExprArrays;
import ru.yandex.commune.mh.builder.MhExprEq;
import ru.yandex.commune.mh.builder.MhExprObjects;
import ru.yandex.commune.mh.builder.MhExprReflect;
import ru.yandex.commune.mh.builder.MhExprThrow;
import ru.yandex.commune.mh.builder.MhFieldRef;
import ru.yandex.commune.mh.builder.MhIfThenElse;
import ru.yandex.misc.lang.Validate;
import ru.yandex.solomon.model.array.mh.compact.CompactArrays;
import ru.yandex.solomon.util.collection.array.ArrayViewMh;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class ArrayCompactFieldMh<A, E> implements ArrayLikeFieldMh<A, E> {

    private final Class<A> arrayClass;
    private final ArrayType<A, E> arrayType;

    private ArrayCompactFieldMh(Class<A> arrayClass, ArrayType<A, E> arrayType) {
        if (arrayClass != arrayType.arrayClass()) {
            throw new IllegalArgumentException("different array classes: " + arrayClass.getSimpleName() + "!=" + arrayType.arrayClass().getSimpleName());
        }

        this.arrayClass = arrayClass;
        this.arrayType = arrayType;
    }

    public ArrayCompactFieldMh(Class<A> arrayClass) {
        this(arrayClass, (ArrayType<A, E>) ArrayType.forArrayClass(arrayClass));
    }

    public ArrayCompactFieldMh(ArrayType<A, E> arrayType) {
        this(arrayType.arrayClass(), arrayType);
    }

    @Override
    public Class<?> arrayLikeType() {
        return Object.class;
    }

    @Override
    public ArrayType<A, E> arrayType() {
        return arrayType;
    }

    @Override
    public MhExpr allocatePrivate(MhExpr capacity) {
        return defaultElementValue().cast(arrayLikeType());
    }

    @Nonnull
    private ArraySimpleFieldMh<A, E> arrayMh() {
        return new ArraySimpleFieldMh<>(arrayType);
    }

    @Nonnull
    private MhExpr copyOfConstants(MhExpr arrayLike, MhExpr newLen) {
        return MhIfThenElse.ifThenElse(
            MhExprEq.eq(newLen, MhConst.intConst(0)),
            empty(),
            arrayLike);
    }

    @Override
    public MhExpr copyOfIfNotEmptyPrivate(MhExpr arrayLike, MhExpr newLen) {
        return switchConstRealNotEmpty(
            arrayLike,
            copyOfConstants(arrayLike, newLen),
            arrayLike,
            MhExprArrays.copyOf(arrayLike.cast(arrayClass), newLen).cast(arrayLikeType())
        );
    }

    @Override
    public MhExpr copyOfPrivate(MhExpr arrayLike, MhExpr newLen) {
        return switchConstRealNotEmpty(
            arrayLike,
            copyOfConstants(arrayLike, newLen),
            defaultElementValue().cast(arrayLikeType()),
            MhExprArrays.copyOf(arrayLike.cast(arrayClass), newLen).cast(arrayLikeType())
        );
    }

    @Override
    public MhExpr copyOfToRegularArrayIfNotEmptyPrivate(MhExpr arrayLike, MhExpr newLen) {
        return switchConstRealNotEmpty(
            arrayLike,
            expand(arrayLike, newLen),
            arrayLike.cast(arrayClass),
            MhExprArrays.copyOf(arrayLike.cast(arrayClass), newLen));
    }

    @Override
    public MhExpr arrayIsNotEmptyPrivate(MhExpr arrayLike) {
        return switchConstRealNotEmpty(
            arrayLike,
            MhConst.boolConst(true),
            MhConst.boolConst(false),
            MhConst.boolConst(true)
        );
    }

    @Override
    public MhExpr arrayLengthPrivate(MhExpr arrayLike) {
        throw new RuntimeException("length is not defined for compact arrays");
    }

    @Override
    public MhExpr swapPrivate(MhExpr arrayLike, MhExpr a, MhExpr b) {
        return onlyIfReal(
            arrayLike,
            MhExprArrays.swap(arrayLike.cast(arrayClass), a, b));
    }

    @Override
    public MhExpr swapIfNotEmptyPrivate(MhExpr arrayLike, MhExpr a, MhExpr b) {
        return onlyIfRealNotEmpty(
            arrayLike,
            MhExprArrays.swap(arrayLike.cast(arrayClass), a, b));
    }

    @Override
    public MethodHandle onlyIfArrayIsNotEmptyPrivate(MethodHandle methodHandle, int arrayPos) {
        throw new RuntimeException("unreachable");
    }

    @Override
    public MhExpr copyElementWithinArrayPrivate(MhExpr arrayLike, MhExpr dst, MhExpr src) {
        return onlyIfReal(
            arrayLike,
            MhExprArrays.copyElement(arrayLike.cast(arrayClass), dst, src));
    }

    @Override
    public MhExpr arrayCopyWithinArrayIfNotEmptyPrivate(MhExpr array, MhExpr src, MhExpr dst, MhExpr count) {
        return onlyIfRealNotEmpty(
            array,
            MhExprArrays.arrayCopyWithinArray(array.cast(arrayClass), src, dst, count));
    }

    @Override
    public MhExpr copyElementWithinArrayIfNotEmptyPrivate(MhExpr arrayLike, MhExpr dst, MhExpr src) {
        return onlyIfRealNotEmpty(
            arrayLike,
            MhExprArrays.copyElement(arrayLike.cast(arrayClass), dst, src));
    }

    @Override
    public MhExpr getPrivate(MhExpr arrayLike, MhExpr index) {
        return switchConstReal(
            arrayLike,
            arrayLike.cast(arrayType.elementClass()),
            MhExprArrays.get(arrayLike.cast(arrayClass), index));
    }

    @Override
    public MhExpr getIfNotEmptyOrDefaultPrivate(MhExpr arrayLike, MhExpr pos) {
        MhExpr getC = arrayLike.cast(arrayType.elementClass());
        MhExpr getA = MhExprArrays.get(arrayLike.cast(arrayClass), pos);
        MhExpr getE = defaultElementValue();
        return switchConstRealNotEmpty(arrayLike, getC, getE, getA);
    }

    private MhExpr expand(MhExpr arrayLike, MhExpr len, MhExpr dst, MhExpr value) {
        MhExpr consts = arrayLike.cast(arrayType.elementClass());
        MhExpr newArray = MhExprArrays.newArray(arrayClass, len);
        // TODO: fill only if value != consts
        MhExpr fill = MhExprArrays.fill(newArray, consts).andReturnParam(0);
        MhExpr set = MhExprArrays.set(fill, dst, value).andReturnParam(0);
        return set.cast(arrayLikeType());
    }

    private MhExpr expand(MhExpr arrayLike, MhExpr len) {
        MhExpr consts = arrayLike.cast(arrayType.elementClass());
        return MhExprArrays.newArrayFilled(arrayClass, len, consts);
    }

    private MhExpr setAndReturnConsts(MhExpr arrayLike, MhExpr len, MhExpr dst, MhExpr value) {
        MhExpr consts = arrayLike.cast(arrayType.elementClass());
        MhExpr eq = MhExprEq.eq(consts, value);
        return MhIfThenElse.ifThenElse(
            eq,
            arrayLike,
            expand(arrayLike, len, dst, value));
    }

    private MhExpr setAndReturnReal(MhExpr arrayLike, MhExpr len, MhExpr dst, MhExpr value) {
        return MhExprArrays.set(arrayLike, dst, value).andReturn(arrayLike);
    }

    @Override
    public MhExpr setAndReturnPrivate(MhExpr arrayLike, MhExpr len, MhExpr dst, MhExpr value) {
        return switchConstReal(
            arrayLike,
            setAndReturnConsts(arrayLike, len, dst, value),
            setAndReturnReal(arrayLike.cast(arrayClass), len, dst, value).cast(arrayLikeType())
        );
    }

    private MhExpr setInFieldConst(MhFieldRef field, MhExpr len, MhExpr dst, MhExpr value) {
        MhExpr arrayLike = field.get();
        MhExpr consts = arrayLike.cast(arrayType.elementClass());
        MhExpr eq = MhExprEq.eq(consts, value);
        return MhIfThenElse.ifThenElse(
            eq,
            MhConst.voidConst(),
            field.set(expand(arrayLike, len, dst, value)));
    }

    @Override
    public MhExpr setInFieldPrivate(MhFieldRef field, MhExpr len, MhExpr dst, MhExpr value) {
        MhExpr arrayLike = field.get();
        return switchConstReal(
            arrayLike,
            setInFieldConst(field, len, dst, value),
            MhExprArrays.set(arrayLike.cast(arrayClass), dst, value)
        );
    }

    @Override
    public MhExpr setInFieldIfNotEmptyPrivate(MhFieldRef field, MhExpr len, MhExpr dst, MhExpr value) {
        MhExpr arrayLike = field.get();
        // TODO: could do better
        return onlyIfNotEmptyAny(
            arrayLike,
            setInFieldPrivate(field, len, dst, value));
    }

    @Override
    public MhExpr setInFieldSingleIfNotEmptyPrivate(MhFieldRef field, MhExpr value) {
        MhExpr arrayLike = field.get();
        return onlyIfNotEmptyAny(
            arrayLike,
            field.set(value.cast(arrayLikeType())));
    }

    private MhExpr realEqualToConstants(MhExpr realArray, MhExpr from, MhExpr to, MhExpr constants) {
        realArray = realArray.cast(arrayClass);
        constants = constants.cast(arrayType.elementClass());
        MhExpr arrayView = ArrayViewMh.wrapArray(realArray, from, to);
        return ArrayViewMh.allAre(arrayView, constants);
    }

    private MhExpr constsEqual(MhExpr c1, MhExpr c2) {
        c1 = c1.cast(arrayType.elementClass());
        c2 = c2.cast(arrayType.elementClass());
        return MhExprEq.eq(c1, c2);
    }

    @Override
    public MhExpr equalPrefixesPrivate(MhExpr a1, MhExpr a2, MhExpr prefix) {
        return switchConstReal(
            a1,
            // a1 is const
            switchConstReal(
                a2,
                constsEqual(a1, a2),
                realEqualToConstants(a2, MhConst.intConst(0), prefix, a1)
            ),
            // a1 is real
            switchConstReal(
                a2,
                realEqualToConstants(a1, MhConst.intConst(0), prefix, a2),
                MhExprArrays.equalPrefixes(a1.cast(arrayClass), a2.cast(arrayClass), prefix)));
    }

    @Override
    public MhExpr equalRangesPrivate(MhExpr a, MhExpr aFrom, MhExpr aTo, MhExpr b, MhExpr bFrom, MhExpr bTo) {
        if (!arrayType().elementClass().isPrimitive()) {
            return MhExprThrow.throwException(boolean.class, RuntimeException.class, "notImplemented");
        }

        return switchConstReal(
            a,
            // a is const
            switchConstReal(
                b,
                constsEqual(a, b),
                realEqualToConstants(b, bFrom, bTo, a)
            ),
            // a is real
            switchConstReal(
                b,
                realEqualToConstants(a, aFrom, aTo, b),
                MhExprArrays.equalRanges(
                    a.cast(arrayClass), aFrom, aTo,
                    b.cast(arrayClass), bFrom, bTo)
            )
        );
    }

    @Override
    public MhExpr hashCodeOfRangePrivate(MhExpr a, MhExpr from, MhExpr to) {
        return switchConstReal(
            a,
            CompactArrays.hashCodeOfRangeConstant(a, from, to),
            MhExprArrays.hashCodeOfRange(a.cast(arrayClass), from, to)
        );
    }

    @Override
    public MhExpr clonePrivate(MhExpr arrayLike) {
        return switchConstRealNotEmpty(arrayLike,
            arrayLike,
            arrayLike,
            MhExprArrays.copyOf(arrayLike.cast(arrayClass)).cast(arrayLikeType())
        );
    }

    @Override
    public MhExpr toStringRangePrivate(MhExpr a, MhExpr from, MhExpr to) {
        return switchConstReal(
            a,
            MhExprObjects.toString(a),
            MhExprArrays.toStringOfRange(a.cast(arrayClass), from, to)
        );
    }

    @Nonnull
    private MhExpr arrayIsPrimitiveWrapper(MhExpr array) {
        return MhExprReflect.objectClassIs(array, arrayType.elementWrapperClass());
    }

    @Nonnull
    private MhExpr arrayIsRealArray(MhExpr array) {
        return MhExprReflect.objectClassIs(array, arrayClass);
    }

    @Nonnull
    private MhExpr throwBadArrayTypeExpr(Class<?> returnType, MhExpr arrayLike) {
        MhExpr throwable = MhCall.staticMethod(CompactArrays.class, "unknownArrayType", arrayLike);
        return MhExprThrow.throwException(returnType, throwable);
    }

    @Nonnull
    private MhExpr switchConstReal(MhExpr arrayLike, MhExpr constants, MhExpr realArray) {
        Validate.equals(constants.exprType(), realArray.exprType());

        return MhIfThenElse.ifs(
            Arrays.asList(
                Tuple2.tuple(arrayIsPrimitiveWrapper(arrayLike), constants),
                Tuple2.tuple(arrayIsRealArray(arrayLike), realArray)
            ),
            throwBadArrayTypeExpr(constants.exprType(), arrayLike));
    }

    @Nonnull
    private MhExpr switchConstRealNotEmpty(
        MhExpr arrayLike, MhExpr constants, MhExpr emptyArray, MhExpr notEmptyArray)
    {
        return switchConstReal(
            arrayLike,
            constants,
            MhIfThenElse.ifThenElse(
                MhExprArrays.isNotEmpty(arrayLike.cast(arrayClass)),
                notEmptyArray,
                emptyArray)
        );
    }

    @Nonnull
    private MhExpr onlyIfReal(MhExpr arrayLike, MhExpr real) {
        return switchConstReal(
            arrayLike,
            MhConst.voidConst(),
            real
        );
    }

    @Nonnull
    private MhExpr onlyIfRealNotEmpty(MhExpr arrayLike, MhExpr realNotEmpty) {
        return switchConstRealNotEmpty(
            arrayLike,
            MhConst.voidConst(),
            MhConst.voidConst(),
            realNotEmpty
        );
    }

    @Nonnull
    private MhExpr onlyIfNotEmpty(MhExpr arrayLike, MhExpr consts, MhExpr realNotEmpty) {
        return switchConstRealNotEmpty(
            arrayLike,
            consts,
            MhConst.voidConst(),
            realNotEmpty
        );
    }

    @Nonnull
    private MhExpr onlyIfNotEmptyAny(MhExpr arrayLike, MhExpr expr) {
        // TODO: inefficient
        return onlyIfNotEmpty(arrayLike, expr, expr);
    }
}
