package ru.yandex.solomon.model.array.mh;

import java.lang.invoke.MethodHandle;
import java.lang.reflect.Field;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.type.array.ArrayType;
import ru.yandex.bolts.type.array.PrimitiveArrayType;
import ru.yandex.bolts.type.number.PrimitiveType;
import ru.yandex.commune.mh.builder.MhBuilder;
import ru.yandex.commune.mh.builder.MhCall;
import ru.yandex.commune.mh.builder.MhConst;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.commune.mh.builder.MhFieldRef;
import ru.yandex.misc.lang.Validate;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public interface ArrayLikeFieldMh<A, E> {

    enum WhenNotEmpty {
        WHEN_NOT_EMPTY,
        ALWAYS,
    }

    // Object or array
    Class<?> arrayLikeType();
    ArrayType<A, E> arrayType();

    default MhExpr defaultElementValue() {
        ArrayType<A, E> arrayType = arrayType();
        if (arrayType.elementClass().isPrimitive()) {
            // TODO: for object arrays too
            PrimitiveType<?, ?> elementType =
                PrimitiveArrayType.forPrimitiveArrayClass(arrayType.arrayClass()).elementType();
            return MhConst.constTyped(elementType.defaultValue(), elementType.primitiveClass());
        } else {
            return MhConst.nullOf(arrayType.elementClass());
        }
    }

    MhExpr allocatePrivate(MhExpr capacity);

    default MhExpr allocate(MhExpr capacity) {
        capacity.assertType(int.class);
        return allocatePrivate(capacity).assertType(arrayLikeType());
    }

    default MhExpr empty() {
        return MhConst.constTyped(arrayType().emptyArray(), arrayLikeType());
    }

    // f[], int -> f
    MhExpr getIfNotEmptyOrDefaultPrivate(MhExpr arrayLike, MhExpr pos);

    default MhExpr getIfNotEmptyOrDefault(MhExpr arrayLike, MhExpr pos) {
        arrayLike.assertType(arrayLikeType());
        pos.assertType(int.class);
        return getIfNotEmptyOrDefaultPrivate(arrayLike, pos).assertType(arrayType().elementClass());
    }

    // f[], int -> f
    default MethodHandle getIfNotEmptyOrDefault() {
        return MhBuilder.oneshot2(arrayLikeType(), int.class, this::getIfNotEmptyOrDefault);
    }

    // f[], int -> f[]
    MhExpr copyOfIfNotEmptyPrivate(MhExpr arrayLike, MhExpr newLen);

    default MhExpr copyOfIfNotEmpty(MhExpr arrayLike, MhExpr newLen) {
        arrayLike.assertType(arrayLikeType());
        newLen.assertType(int.class);
        return copyOfIfNotEmptyPrivate(arrayLike, newLen).assertType(arrayLikeType());
    }

    default MethodHandle copyOfIfNotEmpty() {
        return MhBuilder.oneshot2(arrayLikeType(), int.class, this::copyOfIfNotEmpty);
    }

    MhExpr copyOfPrivate(MhExpr arrayLike, MhExpr newLen);

    default MhExpr copyOf(MhExpr arrayLike, MhExpr newLen) {
        arrayLike.assertType(arrayLikeType());
        newLen.assertType(int.class);
        return copyOfPrivate(arrayLike, newLen).assertType(arrayLikeType());
    }

    default MethodHandle copyOf() {
        return MhBuilder.oneshot2(arrayLikeType(), int.class, this::copyOf);
    }

    MhExpr copyOfToRegularArrayIfNotEmptyPrivate(MhExpr arrayLike, MhExpr newLen);

    default MhExpr copyOfToRegularArrayIfNotEmpty(MhExpr arrayLike, MhExpr newLen) {
        arrayLike.assertType(arrayLikeType());
        newLen.assertType(int.class);
        MhExpr r = copyOfToRegularArrayIfNotEmptyPrivate(arrayLike, newLen);
        r.assertType(arrayType().arrayClass());
        return r;
    }

    default MethodHandle copyOfToRegularArrayIfNotEmpty() {
        return MhBuilder.oneshot2(arrayLikeType(), int.class, this::copyOfToRegularArrayIfNotEmpty);
    }

    // f[] -> boolean
    MhExpr arrayIsNotEmptyPrivate(MhExpr arrayLike);

    default MhExpr arrayIsNotEmpty(MhExpr arrayLike) {
        arrayLike.assertType(arrayLikeType());
        return arrayIsNotEmptyPrivate(arrayLike).assertType(boolean.class);
    }

    default MethodHandle arrayIsNotEmpty() {
        return MhBuilder.oneshot1(arrayLikeType(), this::arrayIsNotEmpty);
    }

    MhExpr arrayLengthPrivate(MhExpr arrayLike);

    default MhExpr arrayLength(MhExpr arrayLike) {
        arrayLike.assertType(arrayLikeType());
        return arrayLengthPrivate(arrayLike).assertType(int.class);
    }

    // f[], int, int
    MhExpr swapPrivate(MhExpr arrayLike, MhExpr a, MhExpr b);

    default MhExpr swap(MhExpr arrayLike, MhExpr a, MhExpr b) {
        arrayLike.assertType(arrayLikeType());
        a.assertType(int.class);
        b.assertType(int.class);
        return swapPrivate(arrayLike, a, b).assertType(void.class);
    }

    default MethodHandle swap() {
        return MhBuilder.oneshot3(arrayLikeType(), int.class, int.class, this::swap);
    }

    MhExpr swapIfNotEmptyPrivate(MhExpr arrayLike, MhExpr a, MhExpr b);

    default MhExpr swapIfNotEmpty(MhExpr arrayLike, MhExpr a, MhExpr b) {
        arrayLike.assertType(arrayLikeType());
        a.assertType(int.class);
        b.assertType(int.class);
        return swapIfNotEmptyPrivate(arrayLike, a, b).assertType(void.class);
    }

    @Nonnull
    static <A> ArrayLikeFieldMh<A, ?> forArrayClass(Class<A> arrayClass, boolean compact) {
        if (compact) {
            return new ArrayCompactFieldMh<>(arrayClass);
        } else {
            return new ArraySimpleFieldMh<>(arrayClass);
        }
    }

    @Nonnull
    static <A> ArrayLikeFieldMh<A, ?> forArrayType(ArrayType<A, ?> arrayType, boolean compact) {
        if (compact) {
            return new ArrayCompactFieldMh<>(arrayType);
        } else {
            return new ArraySimpleFieldMh<>(arrayType);
        }
    }

    MethodHandle onlyIfArrayIsNotEmptyPrivate(MethodHandle methodHandle, int arrayPos);

    default MethodHandle onlyIfArrayIsNotEmpty(MethodHandle methodHandle, int arrayPos) {
        Validate.equals(arrayLikeType(), methodHandle.type().parameterType(arrayPos));
        Validate.equals(void.class, methodHandle.type().returnType());

        return onlyIfArrayIsNotEmptyPrivate(methodHandle, arrayPos);
    }

    MhExpr copyElementWithinArrayPrivate(MhExpr arrayLike, MhExpr dst, MhExpr src);

    default MhExpr copyElementWithinArray(MhExpr arrayLike, MhExpr dst, MhExpr src) {
        arrayLike.assertType(arrayLikeType());
        dst.assertType(int.class);
        src.assertType(int.class);
        return copyElementWithinArrayPrivate(arrayLike, dst, src).assertVoid();
    }

    default MethodHandle copyElementWithinArray() {
        return MhBuilder.oneshot3(arrayLikeType(), int.class, int.class, this::copyElementWithinArray);
    }

    // f[], src: int, dst: int, count: int
    MhExpr arrayCopyWithinArrayIfNotEmptyPrivate(MhExpr array, MhExpr src, MhExpr dst, MhExpr count);

    default MhExpr arrayCopyWithinArrayIfNotEmpty(MhExpr array, MhExpr src, MhExpr dst, MhExpr count) {
        array.assertType(arrayLikeType());
        src.assertType(int.class);
        dst.assertType(int.class);
        count.assertType(int.class);
        return arrayCopyWithinArrayIfNotEmptyPrivate(array, src, dst, count).assertVoid();
    }

    MhExpr copyElementWithinArrayIfNotEmptyPrivate(MhExpr arrayLike, MhExpr dst, MhExpr src);

    default MhExpr copyElementWithinArrayIfNotEmpty(MhExpr arrayLike, MhExpr dst, MhExpr src) {
        arrayLike.assertType(arrayLikeType());
        dst.assertType(int.class);
        src.assertType(int.class);
        return copyElementWithinArrayIfNotEmptyPrivate(arrayLike, dst, src).assertVoid();
    }

    MhExpr getPrivate(MhExpr arrayLike, MhExpr index);

    default MhExpr get(MhExpr arrayLike, MhExpr index) {
        arrayLike.assertType(arrayLikeType());
        index.assertType(int.class);
        return getPrivate(arrayLike, index).assertType(arrayType().elementClass());
    }

    default MethodHandle get() {
        return MhBuilder.oneshot2(arrayLikeType(), int.class, this::get);
    }

    default MhExpr getFromFieldPrivate(MhExpr arrays, Field field, MhExpr pos) {
        // TODO: cast is only for tests
        MhExpr arrayLike = MhCall.getInstanceField(arrays, field).cast(arrayLikeType());
        return get(arrayLike, pos);
    }

    default MhExpr getFromField(MhExpr arrays, Field field, MhExpr pos) {
        pos.assertType(int.class);
        return getFromFieldPrivate(arrays, field, pos).assertType(arrayType().elementClass());
    }

    default MethodHandle getFromField(Field field) {
        return MhBuilder.oneshot2(
            field.getDeclaringClass(), int.class,
            (arrays, pos) -> getFromField(arrays, field, pos));
    }

    // f[], length, dst, f -> f[]
    MhExpr setAndReturnPrivate(MhExpr arrayLike, MhExpr len, MhExpr dst, MhExpr value);

    default MhExpr setAndReturn(MhExpr arrayLike, MhExpr len, MhExpr dst, MhExpr value) {
        arrayLike.assertType(arrayLikeType());
        len.assertType(int.class);
        dst.assertType(int.class);
        value.assertType(arrayType().elementClass());
        return setAndReturnPrivate(arrayLike, len, dst, value).assertType(arrayLikeType());
    }

    MhExpr setInFieldPrivate(MhFieldRef field, MhExpr len, MhExpr dst, MhExpr value);

    default MhExpr setInField(MhFieldRef field, MhExpr len, MhExpr dst, MhExpr value) {
        len.assertType(int.class);
        dst.assertType(int.class);
        value.assertType(arrayType().elementClass());
        return setInFieldPrivate(field, len, dst, value).assertVoid();
    }

    MhExpr setInFieldIfNotEmptyPrivate(MhFieldRef field, MhExpr len, MhExpr dst, MhExpr value);

    default MhExpr setInFieldIfNotEmpty(MhFieldRef field, MhExpr len, MhExpr dst, MhExpr value) {
        len.assertType(int.class);
        dst.assertType(int.class);
        value.assertType(arrayType().elementClass());
        return setInFieldIfNotEmptyPrivate(field, len, dst, value).assertVoid();
    }

    MhExpr setInFieldSingleIfNotEmptyPrivate(MhFieldRef field, MhExpr value);

    default MhExpr setInFieldSingleIfNotEmpty(MhFieldRef field, MhExpr value) {
        field.assertType(arrayLikeType());
        value.assertType(arrayType().elementClass());
        return setInFieldSingleIfNotEmptyPrivate(field, value).assertVoid();
    }

    MhExpr equalPrefixesPrivate(MhExpr a1, MhExpr a2, MhExpr prefix);

    default MhExpr equalPrefixes(MhExpr a1, MhExpr a2, MhExpr prefix) {
        a1.assertType(arrayLikeType());
        a2.assertType(arrayLikeType());
        prefix.assertType(int.class);
        return equalPrefixesPrivate(a1, a2, prefix).assertType(boolean.class);
    }

    MhExpr equalRangesPrivate(MhExpr a, MhExpr aFrom, MhExpr aTo, MhExpr b, MhExpr bFrom, MhExpr bTo);

    default MhExpr equalRanges(MhExpr a, MhExpr aFrom, MhExpr aTo, MhExpr b, MhExpr bFrom, MhExpr bTo) {
        return equalRangesPrivate(a, aFrom, aTo, b, bFrom, bTo);
    }

    MhExpr hashCodeOfRangePrivate(MhExpr a, MhExpr from, MhExpr to);

    default MhExpr hashCodeOfRange(MhExpr a, MhExpr from, MhExpr to) {
        return hashCodeOfRangePrivate(a, from, to);
    }

    MhExpr clonePrivate(MhExpr arrayLike);

    default MhExpr clone(MhExpr arrayLike) {
        arrayLike.assertType(arrayLikeType());
        return clonePrivate(arrayLike).assertType(arrayLikeType());
    }

    MhExpr toStringRangePrivate(MhExpr a, MhExpr from, MhExpr to);

    default MhExpr toStringRange(MhExpr a, MhExpr from, MhExpr to) {
        return toStringRangePrivate(a, from, to);
    }

}
