package ru.yandex.solomon.model.array.mh;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.type.array.ArrayType;
import ru.yandex.commune.mh.Mhx;
import ru.yandex.commune.mh.builder.MhConst;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.commune.mh.builder.MhExprArrays;
import ru.yandex.commune.mh.builder.MhFieldRef;
import ru.yandex.commune.mh.builder.MhIfThenElse;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
class ArraySimpleFieldMh<A, E> implements ArrayLikeFieldMh<A, E> {

    private final Class<A> arrayClass;
    private final ArrayType<A, E> arrayType;

    private ArraySimpleFieldMh(Class<A> arrayClass, ArrayType<A, E> arrayType) {
        if (arrayType.arrayClass() != arrayClass) {
            throw new IllegalArgumentException("different array classes: " + arrayClass.getSimpleName() + "!=" + arrayType.arrayClass().getSimpleName());
        }

        this.arrayClass = arrayClass;
        this.arrayType = arrayType;
    }

    public ArraySimpleFieldMh(Class<A> arrayClass) {
        this(arrayClass, (ArrayType<A, E>) ArrayType.forArrayClass(arrayClass));
    }

    public ArraySimpleFieldMh(ArrayType<A, E> arrayType) {
        this(arrayType.arrayClass(), arrayType);
    }

    @Override
    public Class<?> arrayLikeType() {
        return arrayClass;
    }

    @Override
    public ArrayType<A, E> arrayType() {
        return arrayType;
    }

    @Override
    public MhExpr allocatePrivate(MhExpr capacity) {
        return MhExprArrays.newArray(arrayClass, capacity);
    }

    // f[], int -> f
    @Override
    public MhExpr getIfNotEmptyOrDefaultPrivate(MhExpr arrayLike, MhExpr pos) {
        return MhIfThenElse.ifThenElse(
            MhExprArrays.isNotEmpty(arrayLike),
            MhExprArrays.get(arrayLike, pos),
            defaultElementValue());
    }

    // f[], int -> f[]
    @Override
    public MhExpr copyOfIfNotEmptyPrivate(MhExpr arrayLike, MhExpr newLen) {
        return switchEmptyNotEmpty(
            arrayLike,
            MhExprArrays.copyOf(arrayLike, newLen),
            arrayLike);
    }

    @Override
    public MhExpr copyOfPrivate(MhExpr arrayLike, MhExpr newLen) {
        return MhExprArrays.copyOf(arrayLike, newLen);
    }

    @Override
    public MhExpr copyOfToRegularArrayIfNotEmptyPrivate(MhExpr arrayLike, MhExpr newLen) {
        return switchEmptyNotEmpty(
            arrayLike,
            MhExprArrays.copyOf(arrayLike, newLen),
            arrayLike);
    }

    @Override
    public MhExpr arrayIsNotEmptyPrivate(MhExpr arrayLike) {
        return MhExprArrays.isNotEmpty(arrayLike);
    }

    @Override
    public MhExpr arrayLengthPrivate(MhExpr arrayLike) {
        return MhExprArrays.length(arrayLike);
    }

    @Override
    public MhExpr swapPrivate(MhExpr arrayLike, MhExpr a, MhExpr b) {
        return MhExprArrays.swap(arrayLike, a, b);
    }

    @Override
    public MhExpr swapIfNotEmptyPrivate(MhExpr arrayLike, MhExpr a, MhExpr b) {
        return ifNotEmpty(
            arrayLike,
            MhExprArrays.swap(arrayLike, a, b));
    }

    @Override
    public MethodHandle onlyIfArrayIsNotEmptyPrivate(MethodHandle methodHandle, int arrayPos) {
        try {
            MethodHandle nop = Mhx.nop(methodHandle.type().parameterArray());

            MethodHandle testPermuted = MethodHandles.permuteArguments(
                this.arrayIsNotEmpty(),
                methodHandle.type().changeReturnType(boolean.class),
                arrayPos);

            return MethodHandles.guardWithTest(testPermuted, methodHandle, nop);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public MhExpr copyElementWithinArrayPrivate(MhExpr arrayLike, MhExpr dst, MhExpr src) {
        return MhExprArrays.copyElement(arrayLike, dst, src);
    }

    @Override
    public MhExpr arrayCopyWithinArrayIfNotEmptyPrivate(MhExpr array, MhExpr src, MhExpr dst, MhExpr count) {
        return ifNotEmpty(
            array,
            MhExprArrays.arrayCopyWithinArray(array, src, dst, count));
    }

    @Override
    public MhExpr copyElementWithinArrayIfNotEmptyPrivate(MhExpr arrayLike, MhExpr dst, MhExpr src) {
        return ifNotEmpty(
            arrayLike,
            MhExprArrays.copyElement(arrayLike, dst, src));
    }

    @Override
    public MhExpr getPrivate(MhExpr arrayLike, MhExpr index) {
        return MhExprArrays.get(arrayLike, index);
    }

    @Override
    public MhExpr setAndReturnPrivate(MhExpr arrayLike, MhExpr len, MhExpr dst, MhExpr value) {
        MhExpr set = MhExprArrays.set(arrayLike, dst, value);
        return set.andReturn(arrayLike);
    }

    @Override
    public MhExpr setInFieldPrivate(MhFieldRef field, MhExpr len, MhExpr dst, MhExpr value) {
        MhExpr array = field.get();
        // TODO: cast is actually only needed for the test
        return MhExprArrays.set(array.cast(arrayClass), dst, value);
    }

    @Override
    public MhExpr setInFieldIfNotEmptyPrivate(MhFieldRef field, MhExpr len, MhExpr dst, MhExpr value) {
        MhExpr array = field.get();
        return ifNotEmpty(array, MhExprArrays.set(array, dst, value));
    }

    @Override
    public MhExpr setInFieldSingleIfNotEmptyPrivate(MhFieldRef field, MhExpr value) {
        MhExpr array = field.get();
        return ifNotEmpty(array, MhExprArrays.set(array, MhConst.intConst(0), value));
    }

    @Override
    public MhExpr equalPrefixesPrivate(MhExpr a1, MhExpr a2, MhExpr prefix) {
        return MhExprArrays.equalPrefixes(a1, a2, prefix);
    }

    @Override
    public MhExpr equalRangesPrivate(MhExpr a, MhExpr aFrom, MhExpr aTo, MhExpr b, MhExpr bFrom, MhExpr bTo) {
        return MhExprArrays.equalRanges(a, aFrom, aTo, b, bFrom, bTo);
    }

    @Override
    public MhExpr hashCodeOfRangePrivate(MhExpr a, MhExpr from, MhExpr to) {
        return MhExprArrays.hashCodeOfRange(a, from, to);
    }

    @Override
    public MhExpr clonePrivate(MhExpr arrayLike) {
        return MhExprArrays.copyOf(arrayLike);
    }

    @Override
    public MhExpr toStringRangePrivate(MhExpr a, MhExpr from, MhExpr to) {
        return MhExprArrays.toStringOfRange(a, from, to);
    }

    @Nonnull
    private MhExpr switchEmptyNotEmpty(MhExpr array, MhExpr notEmpty, MhExpr empty) {
        return MhIfThenElse.ifThenElse(
            MhExprArrays.isNotEmpty(array),
            notEmpty,
            empty);
    }

    @Nonnull
    private MhExpr ifNotEmpty(MhExpr array, MhExpr notEmpty) {
        return switchEmptyNotEmpty(array, notEmpty, MhConst.voidConst());
    }

}
