package ru.yandex.solomon.model.array.mh.compact;

import java.util.Arrays;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Cf;

/**
 * Compact boolean arrays means all trues and all falses are represented as single true or false.
 *
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class CompactBooleanArray {

    public static final boolean[] empty = Cf.BooleanArray.emptyArray();

    public static boolean lengthIsZero(Object array) {
        if (array.getClass() == Boolean.class) {
            return false;
        }
        if (array.getClass() == boolean[].class) {
            return ((boolean[]) array).length == 0;
        }
        throw CompactArrays.unknownArrayType(array);
    }

    public static boolean isNotEmpty(Object array) {
        return !lengthIsZero(array);
    }

    public static boolean get(Object array, int pos) {
        if (array.getClass() == Boolean.class) {
            return (boolean) array;
        }
        if (array.getClass() == boolean[].class) {
            return ((boolean[]) array)[pos];
        }
        throw CompactArrays.unknownArrayType(array);
    }

    public static boolean getIfNotEmptyOrDefault(Object array, int pos) {
        if (array.getClass() == Boolean.class) {
            return (boolean) array;
        }
        if (array.getClass() == boolean[].class) {
            boolean[] booleans = (boolean[]) array;
            if (booleans.length != 0) {
                return booleans[pos];
            } else {
                return false;
            }
        }
        throw CompactArrays.unknownArrayType(array);
    }

    public static Object copyOfIfNotEmpty(Object array, int length) {
        if (array.getClass() == Boolean.class) {
            return array;
        }
        if (array.getClass() == boolean[].class) {
            boolean[] booleans = (boolean[]) array;
            if (booleans.length != 0) {
                return Arrays.copyOf(booleans, length);
            } else {
                return booleans;
            }
        }
        throw CompactArrays.unknownArrayType(array);
    }

    public static boolean[] toOrAs(Object array, int length) {
        if (array.getClass() == Boolean.class) {
            return filled(length, (boolean) array);
        }
        if (array.getClass() == boolean[].class) {
            return (boolean[]) array;
        }
        throw CompactArrays.unknownArrayType(array);
    }

    @Nonnull
    public static Object set(Object array, int length, int pos, boolean value) {
        if (array.getClass() == Boolean.class) {
            boolean b = (boolean) array;
            if (b == value) {
                return array;
            } else {
                boolean[] r = new boolean[length];
                Arrays.fill(r, b);
                r[pos] = value;
                return r;
            }
        }
        if (array.getClass() == boolean[].class) {
            boolean[] booleans = (boolean[]) array;
            booleans[pos] = value;
            return booleans;
        }
        throw CompactArrays.unknownArrayType(array);
    }

    @Nonnull
    public static Object fill(Object array, int length, int from, int to, boolean value) {
        if (array.getClass() == Boolean.class) {
            boolean b = (boolean) array;
            if (b == value) {
                return array;
            } else {
                boolean[] r = new boolean[length];
                if (value) {
                    Arrays.fill(r, from, to, true);
                } else {
                    Arrays.fill(r, 0, from, true);
                    Arrays.fill(r, to, length, true);
                }
                return r;
            }
        }
        if (array.getClass() == boolean[].class) {
            boolean[] booleans = (boolean[]) array;
            if (booleans.length != length) {
                throw new IllegalArgumentException("different length or boolean arrays: " + booleans.length + "!=" + length);
            }
            Arrays.fill(booleans, from, to, value);
            return booleans;
        }
        throw CompactArrays.unknownArrayType(array);
    }

    public static void swap(Object array, int pos1, int pos2) {
        // all elements are equal, there is nothing to swap
        if (array.getClass() == Boolean.class) {
            return;
        }
        if (array.getClass() == boolean[].class) {
            boolean[] booleans = (boolean[]) array;
            boolean tmp = booleans[pos1];
            booleans[pos1] = booleans[pos2];
            booleans[pos2] = tmp;
            return;
        }
        throw CompactArrays.unknownArrayType(array);
    }

    public static void copy(Object array, int dst, int src) {
        // all elements are equal, there is nothing to copy
        if (array.getClass() == Boolean.class) {
            return;
        }
        if (array.getClass() == boolean[].class) {
            boolean[] booleans = (boolean[]) array;
            booleans[dst] = booleans[src];
            return;
        }
        throw CompactArrays.unknownArrayType(array);
    }

    public static Object copyOf(Object array, int newLength) {
        if (array.getClass() == Boolean.class) {
            return array;
        }
        if (array.getClass() == boolean[].class) {
            boolean[] booleans = (boolean[]) array;
            if (booleans.length == 0) {
                return false;
            } else {
                return Arrays.copyOf(booleans, newLength);
            }
        }
        throw CompactArrays.unknownArrayType(array);
    }

    private static boolean[] filled(int length, boolean value) {
        boolean[] r = new boolean[length];
        if (value) {
            Arrays.fill(r, true);
        }
        return r;
    }

    public static boolean[] copyOfToArray(Object array, int newLength) {
        if (array.getClass() == Boolean.class) {
            boolean b =(boolean) array;
            return filled(newLength, b);
        }
        if (array.getClass() == boolean[].class) {
            return Cf.BooleanArray.copyOf((boolean[]) array, newLength);
        }
        throw CompactArrays.unknownArrayType(array);
    }

    public static Object copyOfForMultiArray(Object array, int newLength) {
        if (array.getClass() == Boolean.class) {
            return array;
        }
        if (array.getClass() == boolean[].class) {
            boolean[] booleans = (boolean[]) array;
            if (booleans.length == 0) {
                return booleans;
            } else {
                return Arrays.copyOf(booleans, newLength);
            }
        }
        throw CompactArrays.unknownArrayType(array);
    }

    public static boolean equalPrefixes(Object a, Object b, int prefix) {
        if (a.getClass() == Boolean.class && b.getClass() == Boolean.class) {
            return a == b;
        }

        for (int i = 0; i < prefix; ++i) {
            if (get(a, i) != get(b, i)) {
                return false;
            }
        }
        return true;
    }

    public static boolean isConst(Object array, boolean test) {
        if (array.getClass() == Boolean.class) {
            return (boolean) array == test;
        }
        if (array.getClass() == boolean[].class) {
            return false;
        }
        throw CompactArrays.unknownArrayType(array);
    }
}
