package ru.yandex.solomon.model.point;

import java.lang.invoke.MethodHandle;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.commune.mh.MhHashCode;
import ru.yandex.commune.mh.builder.MhBuilder;
import ru.yandex.commune.mh.builder.MhCall;
import ru.yandex.commune.mh.builder.MhConst;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.commune.mh.builder.MhExprStrings;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.model.point.column.CountColumn;
import ru.yandex.solomon.model.point.column.HasColumnSet;
import ru.yandex.solomon.model.point.column.HistogramColumn;
import ru.yandex.solomon.model.point.column.LogHistogramColumn;
import ru.yandex.solomon.model.point.column.LongValueColumn;
import ru.yandex.solomon.model.point.column.MergeColumn;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.StockpileColumnSet;
import ru.yandex.solomon.model.point.column.StockpileColumnSetType;
import ru.yandex.solomon.model.point.column.StockpileColumnSetTypeMh;
import ru.yandex.solomon.model.point.column.StockpileColumnsMh;
import ru.yandex.solomon.model.point.column.SummaryDoubleColumn;
import ru.yandex.solomon.model.point.column.SummaryInt64Column;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Stepan Koltsov
 *
 * @see DataPoint
 */
public class AggrPoint extends AggrPointData implements HasColumnSet, HasMutableTs {
    @StockpileColumnSetType
    public int columnSet;

    public AggrPoint() {
    }

    public AggrPoint(int mask) {
        reset(mask);
    }

    public AggrPoint(AggrPoint point) {
        super(point);
        this.columnSet = point.columnSetMask();
    }

    public AggrPoint(
            @StockpileColumnSetType int columnSet,
            long tsMillis,
            double valueNum,
            long valueDenom,
            boolean merge,
            long count,
            long stepMillis,
            @Nullable LogHistogram logHistogram,
            @Nullable Histogram histogram,
            @Nullable SummaryInt64Snapshot summaryInt64,
            @Nullable SummaryDoubleSnapshot summaryDouble,
            long longValue)
    {
        super(tsMillis, valueNum, valueDenom, merge, count, stepMillis, logHistogram, histogram, summaryInt64, summaryDouble, longValue);
        this.columnSet = columnSet;
    }

    public void reset() {
        reset(StockpileColumnSet.empty.columnSetMask());
    }

    public void reset(int columnSet) {
        resetToDefaultPrimitives();
        LogHistogramColumn.recycle(this.logHistogram);
        this.logHistogram = LogHistogramColumn.DEFAULT_VALUE;
        HistogramColumn.recycle(this.histogram);
        this.histogram = HistogramColumn.DEFAULT_VALUE;
        SummaryInt64Column.recycle(this.summaryInt64);
        this.summaryInt64 = SummaryInt64Column.DEFAULT_VALUE;
        SummaryDoubleColumn.recycle(this.summaryDouble);
        this.summaryDouble = SummaryDoubleColumn.DEFAULT_VALUE;
        this.columnSet = columnSet;
    }

    public void resetToDefaultPrimitives() {
        this.tsMillis = TsColumn.DEFAULT_VALUE;
        this.valueNum = ValueColumn.DEFAULT_VALUE;
        this.valueDenom = ValueColumn.DEFAULT_DENOM;
        this.merge = MergeColumn.DEFAULT_VALUE;
        this.count = CountColumn.DEFAULT_VALUE;
        this.stepMillis = StepColumn.DEFAULT_VALUE;
        this.longValue = LongValueColumn.DEFAULT_VALUE;
    }

    @Override
    public void setTsMillis(long tsMillis) {
        TsColumn.validateOrThrow(tsMillis);
        this.tsMillis = tsMillis;
        columnSet |= TsColumn.mask;
    }

    public void setValue(double value) {
        setValue(value, ValueColumn.DEFAULT_DENOM);
    }

    public void setValue(double valueNum, long valueDenom) {
        ValueColumn.validateOrThrow(valueNum, valueDenom);
        this.valueNum = valueNum;
        this.valueDenom = valueDenom;
        columnSet |= ValueColumn.mask;
    }

    public void setMerge(boolean merge) {
        MergeColumn.validateOrThrow(merge);
        this.merge = merge;
        columnSet |= MergeColumn.mask;
    }

    public void setCount(long count) {
        CountColumn.validateOrThrow(count);
        this.count = count;
        columnSet |= CountColumn.mask;
    }

    public void setStepMillis(long stepMillis) {
        StepColumn.validateOrThrow(stepMillis);
        this.stepMillis = stepMillis;
        columnSet |= StepColumn.mask;
    }

    public void setLogHistogram(LogHistogram histogram) {
        this.logHistogram = histogram;
        if (logHistogram != LogHistogramColumn.DEFAULT_VALUE) {
            columnSet |= LogHistogramColumn.mask;
        }
    }

    public void setHistogram(Histogram histogram) {
        this.histogram = histogram;
        if (!Objects.equals(histogram, HistogramColumn.DEFAULT_VALUE)) {
            columnSet |= HistogramColumn.mask;
        }
    }

    public void setSummaryInt64(SummaryInt64Snapshot summary) {
        this.summaryInt64 = summary;
        if (!Objects.equals(summaryInt64, SummaryInt64Column.DEFAULT_VALUE)) {
            columnSet |= SummaryInt64Column.mask;
        }
    }

    public void setSummaryDouble(SummaryDoubleSnapshot summary) {
        this.summaryDouble = summary;
        if (!Objects.equals(summaryDouble, SummaryDoubleColumn.DEFAULT_VALUE)) {
            columnSet |= SummaryDoubleColumn.mask;
        }
    }

    public void setLongValue(long longValue) {
        this.longValue = longValue;
        this.columnSet |= LongValueColumn.mask;
    }

    public void clearFields(int columnSet) {
        clearFieldDataByMask(columnSet);
        this.columnSet &= ~columnSet;
    }

    public void clearField(StockpileColumn column) {
        clearFields(column.mask());
    }



    private static final MethodHandle withMask =
        MhBuilder.oneshot2(
            AggrPoint.class, int.class,
            (p, newMask) -> {
                return AggrPointMh.copyOf(newMask, p.cast(AggrPointData.class));
            });

    public AggrPoint withMask(
        @StockpileColumnSetType int columnSet)
    {
        try {
            return (AggrPoint) withMask.invokeExact(this, columnSet);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public static AggrPointBuilder builder() {
        return new AggrPointBuilder();
    }

    public static AggrPoint fullForTest(long tsMillis, double value, boolean merge, long count) {
        AggrPoint point = new AggrPoint();
        point.setTsMillis(tsMillis);
        point.setValue(value, ValueColumn.DEFAULT_DENOM);
        point.setMerge(merge);
        point.setCount(count);
        return point;
    }

    public static AggrPoint fullForTestWithStep(long tsMillis, double value, boolean merge, int count, long stepMillis) {
        AggrPoint point = new AggrPoint();
        point.setTsMillis(tsMillis);
        point.setValue(value, ValueColumn.DEFAULT_DENOM);
        point.setMerge(merge);
        point.setCount(count);
        point.setStepMillis(stepMillis);
        return point;
    }

    public static AggrPoint shortPoint(long tsMillis, LogHistogram histogram) {
        AggrPoint point = new AggrPoint();
        point.setTsMillis(tsMillis);
        point.setLogHistogram(histogram);
        return point;
    }

    public static AggrPoint shortPoint(long tsMillis, double sum) {
        AggrPoint r = new AggrPoint();
        r.setTsMillis(tsMillis);
        r.setValue(sum, ValueColumn.DEFAULT_DENOM);
        return r;
    }

    public static AggrPoint shortPoint(long tsMillis, double sum, long stepMillis) {
        AggrPoint r = new AggrPoint();
        r.setTsMillis(tsMillis);
        r.setValue(sum, ValueColumn.DEFAULT_DENOM);
        r.setStepMillis(stepMillis);
        return r;
    }

    @Nullable
    public Double getValueDividedOrNull() {
        return hasColumn(StockpileColumn.VALUE) ? getValueDivided() : null;
    }

    @Override
    public int columnSetMask() {
        return columnSet;
    }

    public void copyTo(AggrPoint target) {
        target.columnSet = this.columnSet;
        target.tsMillis = this.tsMillis;
        target.valueNum = this.valueNum;
        target.valueDenom = this.valueDenom;
        target.merge = this.merge;
        target.count = this.count;
        target.stepMillis = this.stepMillis;
        target.logHistogram = LogHistogramColumn.copy(this.logHistogram, target.logHistogram);
        target.histogram = HistogramColumn.copy(this.histogram, target.histogram);
        target.summaryInt64 = SummaryInt64Column.copy(this.summaryInt64, target.summaryInt64);
        target.summaryDouble = SummaryDoubleColumn.copy(this.summaryDouble, target.summaryDouble);
        target.longValue = this.longValue;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof AggrPoint)) return false;

        AggrPoint aggrPoint = (AggrPoint) o;

        if (tsMillis != aggrPoint.tsMillis) return false;
        if (Double.compare(aggrPoint.valueNum, valueNum) != 0) return false;
        if (aggrPoint.valueDenom != valueDenom) return false;
        if (merge != aggrPoint.merge) return false;
        if (count != aggrPoint.count) return false;
        if (stepMillis != aggrPoint.stepMillis) return false;
        if (!Objects.equals(logHistogram, aggrPoint.logHistogram)) return false;
        if (!Objects.equals(histogram, aggrPoint.histogram)) return false;
        if (!Objects.equals(summaryInt64, aggrPoint.summaryInt64)) return false;
        if (!Objects.equals(summaryDouble, aggrPoint.summaryDouble)) return false;
        if (longValue != aggrPoint.longValue) return false;

        return true;
    }

    private static final MethodHandle hashCode = MhHashCode.hashCodeFromFields(AggrPoint.class);

    @Override
    public int hashCode() {
        try {
            return (int) hashCode.invokeExact(this);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle toStringHelper =
        MhBuilder.oneshot2(
            AggrPoint.class, ArrayList.class,
            (point, arrayList) -> {
                MhExpr columnSet = StockpileColumnSetTypeMh.get(point);
                return StockpileColumnsMh.invokeAllExprsIfColumnIn(columnSet, c -> {
                    List<MhExpr> value = c.mh().getField(point.cast(AggrPointData.class));
                    MhExpr valueString = c.mh().toString(value);
                    MhExpr prefix = MhConst.auto(c.nameLc + "=");
                    MhExpr string = MhExprStrings.concat(prefix, valueString);
                    return MhCall.instanceMethod(arrayList, "add", string.cast(Object.class)).cast(void.class);
                });
            });

    @Override
    public String toString() {
        ArrayList<String> strings = new ArrayList<>();
        try {
            toStringHelper.invokeExact(this, strings);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
        return strings.stream().collect(Collectors.joining(", ", "AggrPoint{", "}"));
    }

}
