package ru.yandex.solomon.model.point;

import java.time.Instant;
import java.time.format.DateTimeFormatter;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class AggrPointBuilder {
    private AggrPoint point;

    AggrPointBuilder() {
        this.point = new AggrPoint();
    }

    AggrPointBuilder(AggrPoint point) {
        this.point = point;
    }

    public AggrPointBuilder time(long tsMillis) {
        point.setTsMillis(tsMillis);
        return this;
    }

    /**
     * @param time The string must represent a valid instant in UTC and is parsed using
     *             {@link DateTimeFormatter#ISO_INSTANT}, for example {@code 2077-12-03T10:15:30.00Z}
     */
    public AggrPointBuilder time(String time) {
        return this.time(Instant.parse(time).toEpochMilli());
    }

    public AggrPointBuilder doubleValue(double value) {
        point.setValue(value, ValueColumn.DEFAULT_DENOM);
        return this;
    }

    public AggrPointBuilder doubleValue(double value, long denom) {
        point.setValue(value, denom);
        return this;
    }

    public AggrPointBuilder merged() {
        point.setMerge(true);
        return this;
    }

    public AggrPointBuilder merge(boolean value) {
        point.setMerge(value);
        return this;
    }

    public AggrPointBuilder count(long mergeCount) {
        point.setCount(mergeCount);
        return this;
    }

    public AggrPointBuilder stepMillis(long stepMillis) {
        point.setStepMillis(stepMillis);
        return this;
    }

    public AggrPointBuilder logHistogram(LogHistogram histogram) {
        point.setLogHistogram(histogram);
        return this;
    }

    public AggrPointBuilder histogram(Histogram histogram) {
        point.setHistogram(histogram);
        return this;
    }

    public AggrPointBuilder summary(SummaryDoubleSnapshot summary) {
        point.setSummaryDouble(summary);
        return this;
    }

    public AggrPointBuilder summary(SummaryInt64Snapshot summary) {
        point.setSummaryInt64(summary);
        return this;
    }

    public AggrPointBuilder longValue(long value) {
        point.setLongValue(value);
        return this;
    }

    public AggrPoint build() {
        // copy point state, to able build multiple points via single builder instance
        return point.withMask(point.columnSet);
    }

    public AggrPoint build(int columnSet) {
        return point.withMask(columnSet);
    }
}
