package ru.yandex.solomon.model.point;

import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.summary.ImmutableSummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.ImmutableSummaryInt64Snapshot;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.model.point.column.CountColumnRandomData;
import ru.yandex.solomon.model.point.column.LongValueRandomData;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class AggrPointDataTestSupport {
    public static AggrPoint randomPoint() {
        return randomPoint(randomMetricType());
    }

    public static AggrPoint randomPoint(MetricType type) {
        return randomPoint(randomMask(type), ThreadLocalRandom.current());
    }

    public static AggrPoint randomPoint(int columnSet) {
        return randomPoint(columnSet, ThreadLocalRandom.current());
    }

    public static AggrPoint randomPoint(AggrPoint point, int columnSet) {
        return randomPoint(point, columnSet, ThreadLocalRandom.current());
    }

    public static AggrPoint randomPoint(int columnSet, Random r) {
        AggrPoint point = new AggrPoint();
        return randomPoint(point, columnSet, r);
    }

    public static AggrPoint randomPoint(AggrPoint point, int columnSet, Random r) {
        point.columnSet = columnSet;
        if (StockpileColumn.TS.isInSet(columnSet)) {
            point.tsMillis = TsRandomData.randomTs(r);
        }
        if (StockpileColumn.STEP.isInSet(columnSet)) {
            point.stepMillis = r.nextInt(1000000);
        }
        if (StockpileColumn.MERGE.isInSet(columnSet)) {
            point.merge = r.nextBoolean();
        }
        if (StockpileColumn.COUNT.isInSet(columnSet)) {
            point.count = CountColumnRandomData.randomCount(r);
        }
        if (StockpileColumn.VALUE.isInSet(columnSet)) {
            point.valueNum = ValueRandomData.randomNum(r);
            point.valueDenom = ValueRandomData.randomDenom(r);
        }
        if (StockpileColumn.HISTOGRAM.isInSet(columnSet)) {
            point.histogram = randomHist(r);
        }
        if (StockpileColumn.LOG_HISTOGRAM.isInSet(columnSet)) {
            point.logHistogram = randomLogHist(r);
        }
        if (StockpileColumn.DSUMMARY.isInSet(columnSet)) {
            point.summaryDouble = randomDSummary(r);
        }
        if (StockpileColumn.ISUMMARY.isInSet(columnSet)) {
            point.summaryInt64 = randomISummary(r);
        }
        if (StockpileColumn.LONG_VALUE.isInSet(columnSet)) {
            point.longValue = LongValueRandomData.randomLongValue(r);
        }
        return point;
    }

    public static MetricType randomMetricType() {
        MetricType[] values = MetricType.values();
        return values[ThreadLocalRandom.current().nextInt(1, values.length - 1)];
    }

    public static int randomMask(MetricType type) {
        int required = StockpileColumns.minColumnSet(type);
        int max = StockpileColumns.maxColumnSet(type);
        return Stream.of(StockpileColumn.values())
                .filter(c -> c.isInSet(max))
                .filter(c -> ThreadLocalRandom.current().nextBoolean())
                .mapToInt(StockpileColumn::mask)
                .reduce(required, (left, right) -> left | right);
    }

    public static int randomMask() {
        return Stream.of(StockpileColumn.values())
                .filter(c -> ThreadLocalRandom.current().nextBoolean())
                .mapToInt(StockpileColumn::mask)
                .reduce(0, (left, right) -> left | right);
    }

    public static Histogram randomHist(Random random) {
        double[] bounds = new double[10];
        long[] buckets = new long[10];
        for (int index = 0; index < 9; index++) {
            bounds[index] = (index + 1) * 2;
            buckets[index] = Math.abs(random.nextLong());
        }
        bounds[9] = Histograms.INF_BOUND;
        buckets[9] = random.nextInt(100);
        return Histogram.newInstance(bounds, buckets);
    }

    public static SummaryInt64Snapshot randomISummary(Random random) {
        return new ImmutableSummaryInt64Snapshot(
            random.nextInt(1000) + 1,
            random.nextLong(),
            random.nextLong(),
            random.nextLong());
    }

    public static SummaryDoubleSnapshot randomDSummary(Random random) {
        return new ImmutableSummaryDoubleSnapshot(
            random.nextInt(1000) + 1,
                random.nextDouble(),
                random.nextDouble(),
                random.nextDouble());
    }

    public static LogHistogram randomLogHist(Random r) {
        return LogHistogram.newBuilder()
                .setBuckets(IntStream.range(1, r.nextInt(LogHistogram.DEFAULT_MAX_BUCKET_SIZE))
                    .mapToDouble(index -> ValueRandomData.randomNum(r))
                    .map(Math::abs)
                    .filter(d -> !Double.isInfinite(d) && !Double.isNaN(d))
                    .toArray())
                .setCountZero(r.nextInt(10000))
                .setStartPower((int) Math.round(r.nextDouble() * 200 - 100))
                .build();
    }

}
