package ru.yandex.solomon.model.point;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;

import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Stepan Koltsov
 */
public class DataPoint implements Point {
    private final long tsMillis;
    private final double value;

    public DataPoint(Instant ts, double value) {
        this.tsMillis = ts.toEpochMilli();
        this.value = value;
    }

    public DataPoint(long tsMillis, double value) {
        this.tsMillis = tsMillis;
        this.value = value;
    }

    public static DataPoint point(String time, double value) {
        if (time.length() == 8) {
            // HH:mm:ss format
            LocalDateTime localDateTime = LocalDateTime.of(LocalDate.now(), LocalTime.parse(time));
            long tsMillis = localDateTime.toEpochSecond(ZoneOffset.UTC) * 1000;
            return point(tsMillis, value);
        }
        return point(Instant.parse(time).toEpochMilli(), value);
    }

    public static DataPoint point(long tsMillis, double value) {
        return new DataPoint(tsMillis, value);
    }

    @Override
    public long getTsMillis() {
        return tsMillis;
    }

    @Override
    public double getValue() {
        return value;
    }

    @Override
    public String toString() {
        return "DataPoint{" +
            "tsMillis=" + InstantUtils.formatToMillis(tsMillis) +
            ", value=" + value +
            '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        DataPoint dataPoint = (DataPoint) o;

        if (tsMillis != dataPoint.tsMillis) return false;
        return Double.compare(dataPoint.value, value) == 0;

    }

    @Override
    public int hashCode() {
        int result;
        long temp;
        result = Long.hashCode(tsMillis);
        temp = Double.doubleToLongBits(value);
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        return result;
    }
}
