package ru.yandex.solomon.model.point.column;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.histogram.Arrays;
import ru.yandex.solomon.model.type.Histogram;

/**
 * @author Vladimir Gordiychuk
 */
public class HistogramColumn extends ColumnImplBase {
    public static final HistogramColumn C = new HistogramColumn();
    public static final Histogram DEFAULT_VALUE = null;
    public static final int mask = StockpileColumn.HISTOGRAM.mask();

    private HistogramColumn() {
        super(StockpileColumn.HISTOGRAM);
    }

    @Nullable
    public static Histogram merge(@Nullable Histogram left, @Nullable Histogram right) {
        if (isNullOrEmpty(left)) {
            return right;
        }

        if (isNullOrEmpty(right)) {
            return left;
        }

        if (left.boundsEquals(right)) {
            return mergeSameBounds(left, right);
        }

        return mergeDifferentBounds(left, right);
    }

    private static Histogram mergeSameBounds(@Nonnull Histogram left, @Nonnull Histogram right) {
        Histogram result = Histogram.newInstance();
        result.setDenom(right.getDenom());

        if (left.getDenom() == right.getDenom()) {
            // fast pash
            for (int index = 0; index < left.count(); index++) {
                result.setUpperBound(index, left.upperBound(index));
                result.setBucketValue(index, left.value(index) + right.value(index));
            }
            return result;
        }

        double leftMultiplier = denomMultiplier(right.getDenom(), left.getDenom());
        for (int index = 0; index < left.count(); index++) {
            result.setUpperBound(index, left.upperBound(index));
            long leftValue = Math.round(left.value(index) * leftMultiplier);
            long rightValue = right.value(index);
            result.setBucketValue(index, leftValue + rightValue);
        }
        return result;
    }

    private static Histogram mergeDifferentBounds(Histogram left, Histogram right) {
        // TODO: merge as HDR histogram merge it? org.HdrHistogram.AbstractHistogram#add
        double[] bounds = new double[right.count()];
        long[] buckets = new long[right.count()];
        for (int index = 0; index < right.count(); index++) {
            bounds[index] = right.upperBound(index);
            buckets[index] = right.value(index);
        }

        final double leftMultiply = denomMultiplier(right.getDenom(), left.getDenom());

        for (int index = 0; index < left.count(); index++) {
            final int newIndex = Math.min(Arrays.lowerBound(bounds, left.upperBound(index)), buckets.length - 1);
            long leftValue = Math.round(left.value(index) * leftMultiply);
            buckets[newIndex] = buckets[newIndex] + leftValue;
        }

        return Histogram.newInstance(bounds, buckets).setDenom(right.getDenom());
    }

    public static double denomMultiplier(long leftDenom, long rightDenom) {
        long denomDelta = rightDenom - leftDenom;
        if (denomDelta == 0) {
            return 1.0;
        } else if (denomDelta > 0) {
            return (double) rightDenom / denomDelta;
        } else {
            return (double) -denomDelta / (double) leftDenom;
        }
    }

    public static Histogram copy(@Nullable Histogram target) {
        if (target == null) {
            return null;
        }

        return Histogram.newInstance().copyFrom(target);
    }

    public static Histogram copy(@Nullable Histogram source, @Nullable Histogram target) {
        if (source == null) {
            return null;
        }

        return Histogram.orNew(target).copyFrom(source);
    }

    public static void recycle(@Nullable Histogram target) {
        if (target == null) {
            return;
        }

        target.recycle();
    }

    public static boolean isNullOrEmpty(Histogram histogram) {
        return histogram == null || histogram.count() == 0;
    }

    public static String toString(@Nullable Histogram histogram) {
        return String.valueOf(histogram);
    }
}
