package ru.yandex.solomon.model.point.column;

import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.misc.lang.EnumMapUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.salmon.proto.StockpileCanonicalProto;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public enum StockpileColumn {
    TS(0, TsColumn.class, StockpileCanonicalProto.Column.TS),
    VALUE(1, ValueColumn.class, StockpileCanonicalProto.Column.VALUE),
    MERGE(2, MergeColumn.class, StockpileCanonicalProto.Column.MERGE),
    COUNT(3, CountColumn.class, StockpileCanonicalProto.Column.COUNT),
    STEP(4, StepColumn.class, StockpileCanonicalProto.Column.STEP),
    HISTOGRAM(6, HistogramColumn.class, StockpileCanonicalProto.Column.HISTOGRAM),
    LOG_HISTOGRAM(7, LogHistogramColumn.class, StockpileCanonicalProto.Column.LOG_HISTOGRAM),
    ISUMMARY(8, SummaryInt64Column.class, StockpileCanonicalProto.Column.ISUMMARY),
    DSUMMARY(9, SummaryDoubleColumn.class, StockpileCanonicalProto.Column.DSUMMARY),
    LONG_VALUE(10, LongValueColumn.class, StockpileCanonicalProto.Column.LONG_VALUE),
    ;

    private static final StockpileColumn[] valuesForByNumberLookup = {
            TS,
            VALUE,
            MERGE,
            COUNT,
            STEP,
            null,
            HISTOGRAM,
            LOG_HISTOGRAM,
            ISUMMARY,
            DSUMMARY,
            LONG_VALUE
    };

    public final Class<?> columnClass;
    private final int mask;
    private final int index;
    public final StockpileCanonicalProto.Column canonicalColumn;
    public final String nameLc;

    StockpileColumn(int index, Class<?> columnClass, StockpileCanonicalProto.Column canonicalColumn) {
        this.canonicalColumn = canonicalColumn;
        this.columnClass = columnClass;
        this.mask = 1 << index;
        this.index = index;
        this.nameLc = name().toLowerCase();
    }

    private static final StockpileColumn[] values = values();

    @Nullable
    public static StockpileColumn byNumberOrNull(int number) {
        if (number >= 0 && number < valuesForByNumberLookup.length) {
            return valuesForByNumberLookup[number];
        }
        return null;
    }

    @Nonnull
    public static StockpileColumn byNumberOrThrow(int number) {
        StockpileColumn column = byNumberOrNull(number);
        if (column == null) {
            throw new IllegalArgumentException("unknown column number: " + number);
        }
        return column;
    }

    public int index() {
        return index;
    }

    public int mask() {
        return mask;
    }

    public boolean isInSet(int columnSetMask) {
        return (columnSetMask & mask) != 0;
    }

    @Nonnull
    public static StockpileColumn fromCanonicalColumn(StockpileCanonicalProto.Column canonicalColumn) {
        for (StockpileColumn stockpileColumn : values()) {
            if (stockpileColumn.canonicalColumn == canonicalColumn) {
                return stockpileColumn;
            }
        }

        throw new IllegalStateException("Not found StockpileColumn for canonical column " + canonicalColumn);
    }

    static {
        for (StockpileCanonicalProto.Column column : StockpileCanonicalProto.Column.values()) {
            // check
            fromCanonicalColumn(column);
        }
    }

    @Nonnull
    public StockpileColumnMh mh() {
        return new StockpileColumnMh(this);
    }

    public List<StockpileColumnField> dataFields() {
        return Arrays.stream(StockpileColumnField.values())
            .filter(d -> d.column == this)
            .collect(Collectors.toList());
    }

    @Nonnull
    public StockpileColumnField firstField() {
        return dataFields().iterator().next();
    }

    @Nonnull
    public StockpileColumnField maskField() {
        return firstField();
    }

    @Nonnull
    public StockpileColumnField singleDataField() {
        return dataFields().stream()
            .collect(CollectorsF.single());
    }

    public static <A> EnumMap<StockpileColumn, List<A>> groupByColumns(List<A> list) {
        Validate.equals(StockpileColumnField.values().length, list.size());

        return EnumMapUtils.fill(StockpileColumn.class, c -> {
            return c.dataFields().stream()
                .map(d -> list.get(d.ordinal()))
                .collect(Collectors.toList());
        });
    }
}
