package ru.yandex.solomon.model.point.column;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.commune.mh.builder.MhConst;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.commune.mh.builder.MhFieldRef;
import ru.yandex.commune.mh.builder.MhIfThenElse;
import ru.yandex.commune.mh.builder.MhSeq;
import ru.yandex.commune.mh.builder.MhValueTarget;
import ru.yandex.misc.lang.Validate;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileColumnFieldMh {

    public final StockpileColumnField dataField;

    public StockpileColumnFieldMh(StockpileColumnField dataField) {
        this.dataField = dataField;
    }

    @Nonnull
    public static MhExpr invokeAllExprs(Function<StockpileColumnField, MhExpr> f) {
        return MhSeq.seq(Arrays.stream(StockpileColumnField.values()).map(f::apply).toArray(MhExpr[]::new));
    }

    public static Class<?>[] columnPrimitiveTypes() {
        return Arrays.stream(StockpileColumnField.values())
                    .map(c -> c.arrayType.elementClass())
                    .toArray(Class<?>[]::new);
    }

    public static Class<?>[] maskAndColumnPrimitiveTypes() {
        ArrayList<Class<?>> r = new ArrayList<>();
        r.add(StockpileColumnSetMh.primitiveParamType);
        r.addAll(Arrays.asList(columnPrimitiveTypes()));
        return r.toArray(new Class[0]);
    }

    public static Class<?>[] columnArrays() {
        return Arrays.stream(StockpileColumnField.values())
            .map(StockpileColumnField::arrayClass)
            .toArray(Class<?>[]::new);
    }

    public static Class<?>[] columnArraysOrCompact() {
        return Arrays.stream(StockpileColumnField.values())
            .map(StockpileColumnField::arrayOrCompactClass)
            .toArray(Class<?>[]::new);
    }

    @Nonnull
    public static MhExpr copyFields(MhExpr source, MhExpr target) {
        return invokeAllExprs(f -> {
            MhExpr value = f.mh().columnGetter(source);
            return f.mh().columnSetter(target, value);
        });
    }

    @Nonnull
    public static MhExpr copyFieldsFilterByMask(MhExpr source, MhExpr target, MhExpr mask) {
        return invokeAllExprs(f -> {
            MhExpr value = f.mh().columnGetter(source);
            MhExpr set = f.mh().columnSetter(target, value);
            MhExpr cond = f.column.mh().isInSet(mask);
            return set.onlyIf(cond);
        });
    }

    @Nonnull
    public MhExpr defaultValue() {
        return MhConst.constTyped(
            dataField.defaultValue(),
            dataField.arrayType.elementClass());
    }

    @Nonnull
    public Field fieldForColumn(Class<?> dataClass) {
        try {
            Field field = Arrays.stream(dataClass.getDeclaredFields())
                .filter(f -> {
                    StockpileColumnField.A a = f.getAnnotation(StockpileColumnField.A.class);
                    return a != null && a.value() == dataField;
                }).collect(CollectorsF.single());
            field.setAccessible(true);
            return field;
        } catch (Exception e) {
            throw new RuntimeException("failed to find field for column: " + dataField + " in " + dataClass, e);
        }
    }

    @Nonnull
    public MhFieldRef fieldForColumn(MhExpr object) {
        return new MhFieldRef(object, fieldForColumn(object.exprType()));
    }

    public static List<MhFieldRef> fields(MhExpr object) {
        return Arrays.stream(StockpileColumnField.values())
            .map(f -> f.mh().fieldForColumn(object))
            .collect(Collectors.toList());
    }

    public static List<MhExpr> getFieldsRaw(MhExpr object) {
        return Arrays.stream(StockpileColumnField.values())
            .map(f -> f.mh().columnGetter(object))
            .collect(Collectors.toList());
    }

    public static MhExpr reset(MhExpr object) {
        return invokeAllExprs(f -> {
            return f.mh().fieldForColumn(object).set(f.mh().defaultValue());
        });
    }

    public static List<MhExpr> getFieldsFilterByMask(MhExpr object, MhExpr mask) {
        return Arrays.stream(StockpileColumnField.values())
            .map(f -> {
                return MhIfThenElse.ifThenElse(
                    f.column.mh().isInSet(mask),
                    f.mh().columnGetter(object),
                    f.mh().defaultValue()
                );
            })
            .collect(Collectors.toList());
    }

    public static MhExpr setFields(MhExpr object, List<MhExpr> values) {
        Validate.equals(StockpileColumnField.values().length, values.size());

        return invokeAllExprs(d -> {
            return d.mh().columnSetter(object, values.get(d.ordinal()));
        });
    }

    public static List<MhValueTarget> setFields(MhExpr object) {
        return Arrays.stream(StockpileColumnField.values())
            .map(d -> d.mh().columnSetterTarget(object))
            .collect(Collectors.toList());
    }

    @Nonnull
    public MhExpr columnSetter(MhExpr target, MhExpr value) {
        return fieldForColumn(target).set(value);
    }

    @Nonnull
    public MhValueTarget columnSetterTarget(MhExpr target) {
        return fieldForColumn(target);
    }

    @Nonnull
    public MhExpr columnGetter(MhExpr source) {
        return fieldForColumn(source).get();
    }

}
