package ru.yandex.solomon.model.point.column;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.commune.mh.builder.MhCall;
import ru.yandex.commune.mh.builder.MhConst;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.commune.mh.builder.MhSeq;
import ru.yandex.misc.lang.Validate;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileColumnMh {

    private final StockpileColumn column;

    public StockpileColumnMh(StockpileColumn column) {
        this.column = column;
    }

    @Nonnull
    public List<MhExpr> merge(List<MhExpr> a, List<MhExpr> b) {
        Validate.equals(column.dataFields().size(), a.size());
        Validate.equals(column.dataFields().size(), b.size());

        return column.dataFields().stream()
            .map(c -> {
                String suffix = column.dataFields().size() != 1 ? "" + c.indexInColumn() : "";

                ArrayList<MhExpr> args = new ArrayList<>();
                args.addAll(a);
                args.addAll(b);

                return MhCall.staticMethod(column.columnClass, "merge" + suffix, args)
                    .assertType(c.arrayType.elementClass());
            })
            .collect(Collectors.toList());
    }

    @Nonnull
    public MhExpr toString(List<MhExpr> value) {
        return MhCall.staticMethod(column.columnClass, "toString", value);
    }

    private static boolean isInSetImpl(int columnMask, int columnSetMask) {
        return (columnMask & columnSetMask) != 0;
    }

    @Nonnull
    public MhExpr isInSet(MhExpr columnSetMask) {
        columnSetMask.assertType(StockpileColumnSetTypeMh.primitiveClass);

        return MhCall.staticMethod(
            StockpileColumnMh.class,
            "isInSetImpl",
            MhConst.intConst(column.mask()), columnSetMask);
    }

    @Nonnull
    public List<MhExpr> defaultValueProper() {
        return column.dataFields().stream()
            .map(f -> f.mh().defaultValue())
            .collect(Collectors.toList());
    }

    public MhExpr setField(MhExpr target, List<MhExpr> fieldValue) {
        List<StockpileColumnField> dataFields = column.dataFields();
        Validate.equals(dataFields.size(), fieldValue.size());

        return MhSeq.seq(
            IntStream.range(0, dataFields.size())
                .mapToObj(i -> dataFields.get(i).mh().columnSetter(target, fieldValue.get(i)))
                .toArray(MhExpr[]::new));
    }

    public List<MhExpr> getField(MhExpr target) {
        return column.dataFields().stream()
            .map(c -> c.mh().columnGetter(target))
            .collect(Collectors.toList());
    }

    public MhExpr invokeAllFields(Function<StockpileColumnField, MhExpr> exprs) {
        return MhSeq.seq(column.dataFields().stream().map(exprs).collect(Collectors.toList()));
    }
}
