package ru.yandex.solomon.model.point.column;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.salmon.proto.StockpileCanonicalProto;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class StockpileColumnSet implements HasColumnSet {

    public static final StockpileColumnSet empty = new StockpileColumnSet(0);

    private final int mask;

    public StockpileColumnSet(int mask) {
        this.mask = mask;
    }

    @Nonnull
    public StockpileColumnSet withoutColumn(StockpileColumn column) {
        return new StockpileColumnSet(mask & ~column.mask());
    }

    @Nonnull
    public static StockpileColumnSet fromColumns(List<StockpileColumn> columns) {
        int mask = columns.stream()
            .mapToInt(StockpileColumn::mask)
            .reduce(0, (a, b) -> a | b);
        return new StockpileColumnSet(mask);
    }

    @Nonnull
    public static StockpileColumnSet fromColumnsVa(StockpileColumn... columns) {
        return fromColumns(Arrays.asList(columns));
    }

    @Nonnull
    public static StockpileColumnSet fromCanonicalColumns(List<StockpileCanonicalProto.Column> canonicalColumns) {
        List<StockpileColumn> columns = canonicalColumns.stream()
            .map(StockpileColumn::fromCanonicalColumn)
            .collect(Collectors.toList());
        return fromColumns(columns);
    }

    @Override
    public int columnSetMask() {
        return mask;
    }

    @Override
    public String toString() {
        return IntStream.range(0, 32)
            .filter(i -> (mask & (1 << i)) != 0)
            .mapToObj(i -> {
                StockpileColumn column = StockpileColumn.byNumberOrNull(i);
                if (column != null) {
                    return column.name();
                } else {
                    return Integer.toString(i);
                }
            })
            .collect(Collectors.joining(", ", "{", "}"));
    }

    public static String toString(int columnSet) {
        return new StockpileColumnSet(columnSet).toString();
    }


    public static final int maxMask = Arrays.stream(StockpileColumn.values())
        .mapToInt(StockpileColumn::mask)
        .reduce(0, (a, b) -> a | b);

    public static void validate(int mask) {
        if (mask < 0 || mask > maxMask) {
            throw new IllegalArgumentException("Specified mask not valid: " + mask);
        }
    }

    public static boolean needToAddAtLeastOneColumn(int currentMask, int updatedColumnSet) {
        return (currentMask & updatedColumnSet) != updatedColumnSet;
    }

    public static boolean needToAddAtLeastOneColumn(StockpileColumnSet current, StockpileColumnSet updated) {
        return needToAddAtLeastOneColumn(current.mask, updated.mask);
    }

    public List<StockpileColumn> columns() {
        return Arrays.stream(StockpileColumn.values())
            .filter(this::hasColumn)
            .collect(Collectors.toList());
    }

    public List<StockpileCanonicalProto.Column> canonicalColumns() {
        return columns()
            .stream().map(c -> c.canonicalColumn)
            .collect(Collectors.toList());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        StockpileColumnSet that = (StockpileColumnSet) o;

        return mask == that.mask;

    }

    @Override
    public int hashCode() {
        return mask;
    }
}
