package ru.yandex.solomon.model.point.column;

import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.util.collection.enums.EnumMapToInt;

/**
 * @author Vladimir Gordiychuk
 */
public final class StockpileColumns {
    private static final EnumMapToInt<MetricType> MIN_MASK;
    private static final EnumMapToInt<MetricType> MAX_MASK;
    static {
        EnumMapToInt<MetricType> min = new EnumMapToInt<>(MetricType.class);
        min.set(MetricType.DGAUGE, StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask());
        min.set(MetricType.IGAUGE, StockpileColumn.TS.mask() | StockpileColumn.LONG_VALUE.mask());
        min.set(MetricType.COUNTER, StockpileColumn.TS.mask() | StockpileColumn.LONG_VALUE.mask());
        min.set(MetricType.RATE, StockpileColumn.TS.mask() | StockpileColumn.LONG_VALUE.mask());
        min.set(MetricType.ISUMMARY, StockpileColumn.TS.mask() | StockpileColumn.ISUMMARY.mask());
        min.set(MetricType.DSUMMARY, StockpileColumn.TS.mask() | StockpileColumn.DSUMMARY.mask());
        min.set(MetricType.HIST, StockpileColumn.TS.mask() | StockpileColumn.HISTOGRAM.mask());
        min.set(MetricType.HIST_RATE, StockpileColumn.TS.mask() | StockpileColumn.HISTOGRAM.mask());
        min.set(MetricType.LOG_HISTOGRAM, StockpileColumn.TS.mask() | StockpileColumn.LOG_HISTOGRAM.mask());
        min.set(MetricType.METRIC_TYPE_UNSPECIFIED, 0);
        MIN_MASK = min;

        EnumMapToInt<MetricType> max = new EnumMapToInt<>(MetricType.class);
        for (MetricType type : MetricType.values()) {
            max.set(type, min.get(type)
                | StockpileColumn.STEP.mask()
                | StockpileColumn.COUNT.mask()
                | StockpileColumn.MERGE.mask());
        }
        MAX_MASK = max;
    }

    private StockpileColumns() {
    }

    public static int minColumnSet(MetricType type) {
        return MIN_MASK.get(type);
    }

    public static int maxColumnSet(MetricType type) {
        return MAX_MASK.get(type);
    }

    public static MetricType typeByMask(int columnSetMask) {
        if (StockpileColumn.VALUE.isInSet(columnSetMask)) {
            return MetricType.DGAUGE;
        }

        if (StockpileColumn.LOG_HISTOGRAM.isInSet(columnSetMask)) {
            return MetricType.LOG_HISTOGRAM;
        }

        if (StockpileColumn.HISTOGRAM.isInSet(columnSetMask)) {
            return MetricType.HIST;
        }

        if (StockpileColumn.ISUMMARY.isInSet(columnSetMask)) {
            return MetricType.ISUMMARY;
        }

        if (StockpileColumn.DSUMMARY.isInSet(columnSetMask)) {
            return MetricType.DSUMMARY;
        }

        if (StockpileColumn.LONG_VALUE.isInSet(columnSetMask)) {
            return MetricType.IGAUGE;
        }

        return MetricType.METRIC_TYPE_UNSPECIFIED;
    }

    public static void ensureColumnSetValid(MetricType type, int mask) {
        int required = minColumnSet(type);
        if ((required & mask) != required) {
            throw new IllegalArgumentException(type
                    + " require column set "
                    + StockpileColumnSet.toString(required)
                    + " but was "
                    + StockpileColumnSet.toString(mask));
        }

        int max = maxColumnSet(type);
        if ((~max & mask) != 0) {
            throw new IllegalArgumentException(type
                    + " support only next column set "
                    + StockpileColumnSet.toString(max)
                    + " but was "
                    + StockpileColumnSet.toString(mask));
        }
    }

    public static void ensureColumnSetValidIfNotEmpty(MetricType type, int mask) {
        if (mask == 0) {
            return;
        }

        if (type == MetricType.METRIC_TYPE_UNSPECIFIED) {
            type = StockpileColumns.typeByMask(mask);
        }

        if (type == MetricType.METRIC_TYPE_UNSPECIFIED) {
            return;
        }

        ensureColumnSetValid(type, mask);
    }
}
