package ru.yandex.solomon.model.point.column;

import java.util.OptionalLong;

import javax.annotation.Nonnull;

import ru.yandex.commune.mh.builder.MhCall;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Stepan Koltsov
 */
public class ValueColumn extends ColumnImplBase {
    private static final long MAX_DENOM = 1L * 1000 * 86400 * 365 * 10;

    public static final ValueColumn C = new ValueColumn();

    public static final StockpileColumn column = StockpileColumn.VALUE;

    public static final double DEFAULT_VALUE = 0;
    public static final long DEFAULT_DENOM = 0;

    public static final int mask = StockpileColumn.VALUE.mask();

    private ValueColumn() {
        super(StockpileColumn.VALUE);
    }

    public static boolean isOne(long denom) {
        return denom == 0 || denom == 1000;
    }

    public static boolean isOneArray(Object array) {
        return array instanceof Long && isOne((Long) array);
    }

    public static OptionalLong getConstArray(Object array) {
        return array instanceof Long ? OptionalLong.of((Long) array) : OptionalLong.empty();
    }

    public static double merge0(double an, long ad, double bn, long bd) {
        validateOrThrow(an, ad);
        validateOrThrow(bn, bd);

        if (Double.isNaN(an) && Double.isNaN(bn)) {
            return Double.NaN;
        }

        if (Double.isNaN(an)) {
            return bn;
        }

        if (Double.isNaN(bn)) {
            return an;
        }

        // TODO: if one of values is NaN, use the other

        if (ad == bd) {
            return an + bn;
        } else {
            return divide(an, ad) + divide(bn, bd);
        }
    }

    public static long merge1(double an, long ad, double bn, long bd) {
        validateOrThrow(an, ad);
        validateOrThrow(bn, bd);

        if (Double.isNaN(an) && Double.isNaN(bn)) {
            return DEFAULT_DENOM;
        }

        if (Double.isNaN(an) && Double.isNaN(bn)) {
            return DEFAULT_DENOM;
        }

        if (Double.isNaN(an)) {
            return bd;
        }

        if (Double.isNaN(bn)) {
            return ad;
        }

        if (ad == bd) {
            return ad;
        } else {
            return DEFAULT_DENOM;
        }
    }

    @Nonnull
    public static ValueObject merge(ValueObject a, ValueObject b) {
        return new ValueObject(
            merge0(a.num, a.denom, b.num, b.denom),
            merge1(a.num, a.denom, b.num, b.denom));
    }

    public static double copy(double num) {
        return num;
    }

    public static double copy(long denom) {
        return denom;
    }

    public static double divide(double num, long denom) {
        validateOrThrow(num, denom);

        if (isOne(denom)) {
            return num;
        } else {
            return num / denom * 1000;
        }
    }

    public static MhExpr divide(MhExpr num, MhExpr denom) {
        return MhCall.staticMethod(ValueColumn.class, "divide", num, denom);
    }

    public static String toString(double num, long denom) {
        String r = Double.toString(num);
        if (!isOne(denom)) {
            r += "/" + DurationUtils.millisToSecondsString(denom);
        }
        return r;
    }

    public static void validateOrThrow(double valueNum, long valueDenom) {
        // probably prohibit 1000 too, because 1000 is the same as 0
        if (valueDenom < 0) {
            throw new RuntimeException("invalid denom: " + toString(valueNum, valueDenom));
        }
        if (valueDenom > MAX_DENOM) {
            throw new RuntimeException("sanity check: " + toString(valueNum, valueDenom));
        }
    }

    public static boolean isValidDenom(long valueDenom) {
        return valueDenom >= 0 && valueDenom < MAX_DENOM;
    }
}
