package ru.yandex.solomon.model.point.column;

import java.util.Arrays;
import java.util.stream.DoubleStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class ValueView {

    private final double[] valuesNum;
    private final Object valuesDenom;
    private final int from;
    private final int to;

    public ValueView(double[] valuesNum, Object valuesDenom, int from, int to) {
        this.valuesNum = valuesNum;
        this.valuesDenom = valuesDenom;
        this.from = from;
        this.to = to;
    }

    public ValueView(double[] values, int from, int to) {
        this(values, ValueColumn.DEFAULT_DENOM, from, to);
    }

    public int length() {
        return to - from;
    }

    private int indexToArrayIndex(int i) {
        Validate.isTrue(i >= 0 && i < length());
        return i + from;
    }

    public double getValueNum(int i) {
        return valuesNum[indexToArrayIndex(i)];
    }

    public long getValueDenom(int i) {
        if (valuesDenom instanceof Long) {
            return (long) valuesDenom;
        }
        if (valuesDenom instanceof long[]) {
            return ((long[]) valuesDenom)[indexToArrayIndex(i)];
        }
        throw new AssertionError();
    }

    public double getValueDivided(int i) {
        return ValueColumn.divide(getValueNum(i), getValueDenom(i));
    }

    public double[] dividedValues() {
        int len = length();
        double[] values = new double[len];
        for (int i = 0; i < len; i++) {
            values[i] = getValueDivided(i);
        }
        return values;
    }

    @Nonnull
    private DoubleStream numStream() {
        return Arrays.stream(valuesNum, from, to);
    }

    @Nonnull
    public ValueObject sum() {
        if (length() == 0) {
            return ValueObject.Zero;
        }

        double num = getValueNum(0);
        long denom = getValueDenom(0);

        for (int i = 1; i < length(); ++i) {
            if (num == 0) {
                // special case to avoid losing of denom
                // if denoms are different
                num = getValueNum(i);
                denom = getValueDenom(i);
            } else if (ValueColumn.isOne(denom)) {
                // denom is lost, so the num is probably a random junk
                // so we won't loose anything using 1 as denom
                num += getValueDivided(i);
            } else if (denom == getValueDenom(i)) {
                // happiest case
                num += getValueNum(i);
            } else {
                // we could probably use LCD here
                num = ValueColumn.divide(num, denom);
                denom = ValueColumn.DEFAULT_DENOM;
                num += getValueDivided(i);
            }
        }

        return new ValueObject(num, denom);
    }

    @Nonnull
    public ValueObject avgSum() {
        // special case
        if (length() == 0) {
            return ValueObject.NaN;
        }

        // optimization
        if (length() == 1) {
            return getValueObject(0);
        }

        ValueObject sum = sum();
        if (sum.isDenomOne()) {
            // TODO
            return new ValueObject(sum.num / length(), ValueColumn.DEFAULT_DENOM);
        } else {
            return new ValueObject(sum.num, sum.denom * length());
        }
    }

    @Nonnull
    public ValueObject getValueObject(int i) {
        return new ValueObject(getValueNum(i), getValueDenom(i));
    }
}
