package ru.yandex.solomon.model.timeseries;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.commune.mh.builder.MhBuilder;
import ru.yandex.commune.mh.builder.MhCall;
import ru.yandex.commune.mh.builder.MhConst;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.commune.mh.builder.MhIfThenElse;
import ru.yandex.misc.algo.arrayList.ArrayListImpl;
import ru.yandex.misc.algo.sort.ArraySortAccessor;
import ru.yandex.misc.algo.sort.StableSortImpl;
import ru.yandex.misc.lang.Validate;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.HasColumnSet;
import ru.yandex.solomon.model.point.column.HistogramColumn;
import ru.yandex.solomon.model.point.column.LogHistogramColumn;
import ru.yandex.solomon.model.point.column.MergeColumn;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.StockpileColumnField;
import ru.yandex.solomon.model.point.column.StockpileColumnFieldMh;
import ru.yandex.solomon.model.point.column.StockpileColumnSet;
import ru.yandex.solomon.model.point.column.StockpileColumnSetType;
import ru.yandex.solomon.model.point.column.StockpileColumnsMh;
import ru.yandex.solomon.model.point.column.StockpileMergeMh;
import ru.yandex.solomon.model.point.column.SummaryDoubleColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.point.column.ValueObject;
import ru.yandex.solomon.model.point.column.ValueView;
import ru.yandex.solomon.model.point.predicate.AggrDataPointPredicate;
import ru.yandex.solomon.model.timeseries.view.DoubleTimeSeriesView;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.model.type.SummaryDouble;
import ru.yandex.solomon.model.type.SummaryInt64;
import ru.yandex.solomon.util.collection.array.LongArrayView;

/**
 * @author Stepan Koltsov
 *
 * @see GraphDataArrayList
 */
@ParametersAreNonnullByDefault
public class AggrGraphDataArrayList extends AggrGraphDataArrayListOrView implements AggrGraphDataSink, Cloneable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(AggrGraphDataArrayList.class);

    public static final AggrGraphDataArraysMh<AggrGraphDataArrayList> arraysMh =
        new AggrGraphDataArraysMh<AggrGraphDataArrayList>(AggrGraphDataArrayList.class, MethodHandles.lookup()) {

            @Nonnull
            @Override
            MhExpr length(MhExpr arrays) {
                return MhCall.getInstanceField(arrays, "size");
            }

            @Override
            MhExpr arraysFrom(MhExpr arrays) {
                return MhConst.intConst(0);
            }

            @Override
            MhExpr arraysTo(MhExpr arrays) {
                return length(arrays);
            }

            @Override
            MhExpr indexToArrayIndex(MhExpr arrays, MhExpr index) {
                return index;
            }
        };


    @StockpileColumnField.A(StockpileColumnField.TS)
    private long[] tss;
    @StockpileColumnField.A(StockpileColumnField.VALUE_NUM)
    private double[] valuesNum;
    @StockpileColumnField.A(StockpileColumnField.VALUE_DENOM)
    private Object valuesDenom;
    @StockpileColumnField.A(StockpileColumnField.MERGE)
    private Object merges;
    @StockpileColumnField.A(StockpileColumnField.COUNT)
    private long[] counts;
    @StockpileColumnField.A(StockpileColumnField.STEP)
    private Object stepMillis;
    @StockpileColumnField.A(StockpileColumnField.LOG_HISTOGRAM)
    private LogHistogram[] logHistograms;
    @StockpileColumnField.A(StockpileColumnField.HISTOGRAM)
    private Histogram[] histograms;
    @StockpileColumnField.A(StockpileColumnField.ISUMMARY)
    private SummaryInt64Snapshot[] summariesInt64;
    @StockpileColumnField.A(StockpileColumnField.DSUMMARY)
    private SummaryDoubleSnapshot[] summariesDouble;
    @StockpileColumnField.A(StockpileColumnField.LONG_VALUE)
    private long[] longValues;
    private int size;

    public AggrGraphDataArrayList(@StockpileColumnSetType int columnSetMask, int capacity) {
        resetWithCap(columnSetMask, capacity);
    }

    public AggrGraphDataArrayList() {
        reset();
    }

    public AggrGraphDataArrayList(
        long[] tss,
        double[] valuesNum,
        Object valuesDenom,
        Object merges,
        long[] counts,
        Object stepMillis,
        LogHistogram[] logHistograms,
        Histogram[] histograms,
        SummaryInt64Snapshot[] summariesInt64,
        SummaryDoubleSnapshot[] summariesDouble,
        long[] longValues)
    {
        this.tss = tss;
        this.valuesNum = valuesNum;
        this.valuesDenom = valuesDenom;
        this.merges = merges;
        this.counts = counts;
        this.stepMillis = stepMillis;
        this.logHistograms = logHistograms;
        this.histograms = histograms;
        this.summariesInt64 = summariesInt64;
        this.summariesDouble = summariesDouble;
        this.longValues = longValues;
        this.size = tss.length;
    }

    public static MhExpr newInstance(List<MhExpr> arrays) {
        Validate.equals(StockpileColumnField.values().length, arrays.size());

        return MhCall.newInstance(AggrGraphDataArrayList.class, arrays);
    }

    public static AggrGraphDataArrayList empty() {
        return new AggrGraphDataArrayList();
    }

    public static AggrGraphDataArrayList listShort(
        long tsMillis, double value)
    {
        AggrGraphDataArrayList r = new AggrGraphDataArrayList();
        r.addRecordShort(tsMillis, value);
        return r;
    }

    public static AggrGraphDataArrayList listShort(
        long tsMillis1, double value1,
        long tsMillis2, double value2)
    {
        AggrGraphDataArrayList r = new AggrGraphDataArrayList();
        r.addRecordShort(tsMillis1, value1);
        r.addRecordShort(tsMillis2, value2);
        return r;
    }

    public static AggrGraphDataArrayList listShort(
        long tsMillis1, double value1,
        long tsMillis2, double value2,
        long tsMillis3, double value3)
    {
        AggrGraphDataArrayList r = new AggrGraphDataArrayList();
        r.addRecordShort(tsMillis1, value1);
        r.addRecordShort(tsMillis2, value2);
        r.addRecordShort(tsMillis3, value3);
        return r;
    }

    public static AggrGraphDataArrayList of(AggrPoint... points) {
        if (points.length == 0) {
            return empty();
        }

        int mask = Stream.of(points)
            .mapToInt(AggrPoint::columnSetMask)
            .reduce((left, right) -> left | right)
            .orElse(0);

        AggrGraphDataArrayList list = new AggrGraphDataArrayList(mask, points.length);

        for (AggrPoint point : points) {
            list.addRecord(point);
        }

        return list;
    }

    public static AggrGraphDataArrayList of(AggrGraphDataIterable source) {
        return of(source.iterator());
    }

    public static AggrGraphDataArrayList of(AggrGraphDataListIterator iterator) {
        int points = iterator.estimatePointsCount();
        int capacity = points == -1 ? 10 : points;
        AggrGraphDataArrayList result = new AggrGraphDataArrayList(iterator.columnSetMask(), capacity);
        result.addAllFrom(iterator);
        return result;
    }

    public static AggrGraphDataArrayList of(DoubleTimeSeriesView view) {
        int mask = StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask();
        AggrGraphDataArrayList result = new AggrGraphDataArrayList(mask, view.length());

        AggrPoint tempPoint = new AggrPoint();
        for (int index = 0; index < view.length(); index++) {
            tempPoint.setTsMillis(view.getTsMillis(index));
            tempPoint.setValue(view.getValue(index), ValueColumn.DEFAULT_DENOM);
            result.addRecordData(mask, tempPoint);
        }

        return result;
    }

    private static final MethodHandle maskFromArrays =
        arraysMh.maskFromArrays();


    @Override
    public int columnSetMask() {
        // should probably return 0 when size is 0
        try {
            return (int) maskFromArrays.invokeExact(this);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle getAnyPoint = arraysMh.makeGetAnyPoint();


    public AggrPoint getAnyPoint(int i) {
        checkIndex(i);

        try {
            return (AggrPoint) getAnyPoint.invokeExact(this, i);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle cloneFields =
        MhBuilder.oneshot3(
            arraysMh.arraysClass, arraysMh.arraysClass, int.class,
            (dst, src, size) -> {
                return StockpileColumnFieldMh.invokeAllExprs(c -> {
                    MhExpr array = arraysMh.getArrayForField(c, src);
                    MhExpr copied = c.arrayLikeMh().copyOfIfNotEmpty(array, size);
                    return arraysMh.setArrayForField(c, dst, copied);
                });
            });

    @Override
    public AggrGraphDataArrayList clone() {
        try {
            AggrGraphDataArrayList r = (AggrGraphDataArrayList) super.clone();
            try {
                cloneFields.invokeExact(r, this, size);
            } catch (Throwable throwable) {
                throw new RuntimeException(throwable);
            }
            return r;
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    private static final MethodHandle cloneWithMask =
        MhBuilder.oneshot3(
            arraysMh.arraysClass, int.class, int.class,
            (list, mask, size) -> {
                List<MhExpr> arrays0 = arraysMh.getArrays(list);
                List<MhExpr> arrays = Arrays.stream(StockpileColumnField.values())
                    .map(d -> {
                        MhExpr array = arrays0.get(d.ordinal());
                        return MhIfThenElse.ifThenElse(
                            d.column.mh().isInSet(mask),
                            d.arrayLikeMh().copyOfIfNotEmpty(array, size),
                            d.arrayLikeMh().empty()
                        );
                    })
                    .collect(Collectors.toList());
                return newInstance(arrays);
            });

    @Nonnull
    public AggrGraphDataArrayList cloneWithMask(int columnSetMask) {
        if (size == 0) {
            return new AggrGraphDataArrayList();
        }

        try {
            return (AggrGraphDataArrayList) cloneWithMask.invokeExact(this, columnSetMask, size);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private int capacity() {
        return tss.length;
    }

    public void clear() {
        if (histograms != null && histograms.length != 0) {
            for (int index = 0; index < size; index++) {
                HistogramColumn.recycle(histograms[index]);
                histograms[index] = null;
            }
        }

        if (logHistograms != null && logHistograms.length != 0) {
            for (int index = 0; index < size; index++) {
                LogHistogramColumn.recycle(logHistograms[index]);
                logHistograms[index] = null;
            }
        }

        if (summariesDouble != null && summariesDouble.length != 0) {
            for (int index = 0; index < size; index++) {
                SummaryDoubleColumn.recycle(summariesDouble[index]);
                summariesDouble[index] = null;
            }
        }
        size = 0;
    }

    private static final MethodHandle reset =
        MhBuilder.oneshot1(
            AggrGraphDataArrayList.class,
            list -> {
                return StockpileColumnFieldMh.invokeAllExprs(c -> {
                    return arraysMh.setArrayForField(c, list, MhConst.auto(c.arrayType.emptyArray()));
                });
            });

    private static final MethodHandle resetWithCap =
        MhBuilder.oneshot3(
            AggrGraphDataArrayList.class, int.class, int.class,
            (list, cap, columnSetMask) -> {
                return StockpileColumnFieldMh.invokeAllExprs(c -> {
                    MhExpr allocated = c.arrayLikeMh().allocate(cap);
                    MhExpr empty = c.arrayLikeMh().empty();
                    MhExpr cond = c.column.mh().isInSet(columnSetMask);
                    MhExpr array = MhIfThenElse.ifThenElse(cond, allocated, empty);
                    return arraysMh.setArrayForField(c, list, array);
                });
            });

    private void reset() {
        try {
            reset.invokeExact(this);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
        size = 0;
    }

    private void resetWithCap(int columnSetMask, int capacity) {
        if (capacity == 0) {
            StockpileColumnSet.validate(columnSetMask);

            reset();
        } else {
            validateColumnSet(columnSetMask);

            try {
                resetWithCap.invokeExact(this, capacity, columnSetMask);
            } catch (Throwable throwable) {
                throw new RuntimeException(throwable);
            }
            this.size = 0;
        }
    }

    private static void validateColumnSet(int columnSetMask) {
        StockpileColumnSet.validate(columnSetMask);
        if (!HasColumnSet.hasColumn(columnSetMask, StockpileColumn.TS)) {
            throw new IllegalArgumentException("must have ts column: " + StockpileColumnSet.toString(columnSetMask));
        }
    }

    public void foldDenomIntoOne() {
        if (isEmpty()) {
            return;
        }

        if (!hasColumn(StockpileColumn.VALUE)) {
            return;
        }

        if (ValueColumn.isOneArray(valuesDenom)) {
            return;
        }

        for (int i = 0; i < size; ++i) {
            valuesNum[i] = getValueDividedOrThrow(i);
        }

        valuesDenom = ValueColumn.DEFAULT_DENOM;
    }

    private int remainingCapacity() {
        return this.capacity() - this.size;
    }

    private void reserveAdditional(int additional, int columnSetMask) {
        if (remainingCapacity() < additional) {
            ArrayListImpl.reserveAdditional(new AccessorImpl(columnSetMask), additional);
        }
    }

    private static final MethodHandle reserveExact =
        MhBuilder.oneshot3(
            AggrGraphDataArrayList.class, int.class, int.class,
            (list, newCapacity, columnSetMask) -> {
                return StockpileColumnFieldMh.invokeAllExprs(c -> {
                    MhExpr array = arraysMh.getArrayForField(c, list);
                    MhExpr newArray = c.arrayLikeMh().copyOf(array, newCapacity);
                    MhExpr set = arraysMh.setArrayForField(c, list, newArray);
                    return set.onlyIf(c.column.mh().isInSet(columnSetMask));
                });
            });

    private void reserveExact(int columnSetMask, int newCapacity) {
        try {
            reserveExact.invokeExact(this, newCapacity, columnSetMask);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public void truncate(int newSize) {
        if (newSize > size) {
            throw new IllegalArgumentException("cannot truncate array with new size " + newSize + " > " + size);
        }
        size = newSize;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        if (tss != null) {
            size += MemoryCounter.arrayObjectSize(tss);
        }
        if (valuesNum != null) {
            size += MemoryCounter.arrayObjectSize(valuesNum);
        }
        if (valuesDenom instanceof long[]) {
            size += MemoryCounter.arrayObjectSize((long[]) valuesDenom);
        }
        if (merges instanceof boolean[]) {
            size += MemoryCounter.arrayObjectSize((boolean[]) merges);
        }
        if (counts != null) {
            size += MemoryCounter.arrayObjectSize(counts);
        }
        if (stepMillis instanceof long[]) {
            size += MemoryCounter.arrayObjectSize((long[]) stepMillis);
        }
        if (logHistograms != null) {
            size += MemoryCounter.arrayObjectSize(logHistograms);
            size += LogHistogram.SELF_SIZE;
        }
        if (histograms != null) {
            size += MemoryCounter.arrayObjectSize(histograms);
            size += Histogram.SELF_SIZE * histograms.length;
        }
        if (summariesInt64 != null) {
            size += MemoryCounter.arrayObjectSize(summariesInt64);
            size += SummaryInt64.SELF_SIZE * summariesInt64.length;
        }
        if (summariesDouble != null) {
            size += MemoryCounter.arrayObjectSize(summariesDouble);
            size += SummaryDouble.SELF_SIZE * summariesDouble.length;
        }
        if (longValues != null) {
            size += MemoryCounter.arrayObjectSize(longValues);
        }
        return size;
    }

    private class AccessorImpl implements ArrayListImpl.Accessor {

        @StockpileColumnSetType
        private final int columnSetMask;

        private AccessorImpl(int columnSetMask) {
            this.columnSetMask = columnSetMask;
        }

        @Override
        public int capacity() {
            return AggrGraphDataArrayList.this.capacity();
        }
        @Override
        public int size() {
            return size;
        }

        @Override
        public void reserveExact(int newCapacity) {
            AggrGraphDataArrayList.this.reserveExact(columnSetMask, newCapacity);
        }
    }

    private static final MethodHandle copyToSize =
        MhBuilder.oneshot2(
            AggrGraphDataArrayList.class, int.class,
            (list, size) -> {
                return StockpileColumnFieldMh.invokeAllExprs(c -> {
                    MhExpr array = arraysMh.getArrayForField(c, list);
                    MhExpr newArray = c.arrayLikeMh().copyOfIfNotEmpty(array, size);
                    return arraysMh.setArrayForField(c, list, newArray);
                });
            });

    public void trimToSize() {
        if (size != tss.length) {
            try {
                copyToSize.invokeExact(this, size);
            } catch (Throwable throwable) {
                throw new RuntimeException(throwable);
            }
        }
    }


    private static final MethodHandle expandToColumnSetHelper =
        MhBuilder.oneshot3(
            AggrGraphDataArrayList.class, int.class, int.class,
            (list, capacity, columnsToAllocate) -> {
                return StockpileColumnsMh.invokeAllExprsIfColumnIn(columnsToAllocate, c -> {
                    return c.mh().invokeAllFields(d -> {
                        return arraysMh.setArrayForField(d, list, d.arrayLikeMh().allocate(capacity));
                    });
                });
            });


    private void expandToColumnSet(int newColumnSetMask) {
        StockpileColumnSet.validate(newColumnSetMask);

        int currentColumnSetMask = columnSetMask();
        if (!StockpileColumnSet.needToAddAtLeastOneColumn(currentColumnSetMask, newColumnSetMask)) {
            return;
        }

        if (isEmpty() && capacity() == 0) {
            return;
        }

        {
            int resultingColumns = newColumnSetMask | currentColumnSetMask;
            validateColumnSet(resultingColumns);
        }

        int columnsToAllocate = newColumnSetMask & ~currentColumnSetMask;

        try {
            expandToColumnSetHelper.invokeExact(this, tss.length, columnsToAllocate);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    @Override
    public void ensureCapacity(int columnSet, int minCapacity) {
        int expandedColumnSet = columnSet | columnSetMask();
        expandToColumnSet(expandedColumnSet);
        if (minCapacity > capacity()) {
            reserveExact(expandedColumnSet, minCapacity);
        }
    }

    @Override
    public void addRecordData(@StockpileColumnSetType int columnSet, AggrPointData data) {
        expandToColumnSet(columnSet);

        reserveAdditional(1, this.columnSetMask() | columnSet);

        ++size;
        setUnchecked(size - 1, data);
    }

    @Override
    public void addAllFrom(AggrGraphDataListIterator iterator) {
        int columnSetMask = iterator.columnSetMask() | this.columnSetMask();

        var point = RecyclableAggrPoint.newInstance();
        try {
            if (iterator.next(point)) {
                expandToColumnSet(columnSetMask);
            } else {
                return;
            }

            do {
                if (size == capacity()) {
                    // TODO: could use size hint here
                    reserveAdditional(1, columnSetMask);
                }

                setUnchecked(size++, point);
            } while (iterator.next(point));
        } finally {
            point.recycle();
        }
    }

    public void addRecordFullForTest(long ts, double value, boolean merge, int count, long stepMillis) {
        addRecord(AggrPoint.fullForTestWithStep(ts, value, merge, count, stepMillis));
    }

    public void addRecordFullForTest(long ts, double value, boolean merge, long count) {
        addRecord(AggrPoint.fullForTest(ts, value, merge, count));
    }

    public void addRecordShort(long ts, double value) {
        addRecord(AggrPoint.shortPoint(ts, value));
    }

    private static final MethodHandle setRawFromData =
        MhBuilder.oneshot3(
            AggrGraphDataArrayList.class, int.class, AggrPointData.class,
            (arrays, pos, aggrPointData) -> {
                return StockpileColumnFieldMh.invokeAllExprs(c -> {
                    final MhExpr value = c.mh().columnGetter(aggrPointData);
                    final MhExpr copy;
                    if (c.mutable) {
                        copy = MhCall.staticMethod(c.column.columnClass, "copy", value)
                            .assertType(c.arrayType.elementClass());
                    } else {
                        copy = value;
                    }

                    MhExpr capacity = arraysMh.capacity(arrays);

                    return c.arrayLikeMh().setInFieldIfNotEmpty(
                        arraysMh.fieldRef(c, arrays),
                        capacity,
                        pos,
                        copy);
                });
            });

    private static final MethodHandle setRawFromDataWhenFirst =
        MhBuilder.oneshot2(
            AggrGraphDataArrayList.class, AggrPointData.class,
            (arrays, aggrPointData) -> {
                return StockpileColumnFieldMh.invokeAllExprs(c -> {
                    final MhExpr value = c.mh().columnGetter(aggrPointData);
                    final MhExpr copy;
                    if (c.mutable) {
                        copy = MhCall.staticMethod(c.column.columnClass, "copy", value)
                            .assertType(c.arrayType.elementClass());
                    } else {
                        copy = value;
                    }
                    return c.arrayLikeMh().setInFieldSingleIfNotEmpty(
                        arraysMh.fieldRef(c, arrays),
                        copy);
                });
            });

    private void setUnchecked(int index, AggrPointData data) {
        try {
            if (index == 0 && size == 1) {
                // special case for better handling of compact arrays
                setRawFromDataWhenFirst.invokeExact(this, data);
            } else {
                setRawFromData.invokeExact(this, index, data);
            }
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public void setData(int index, AggrPointData data) {
        checkIndex(index);
        setUnchecked(index, data);
    }

    private void setLast(AggrPointData point) {
        setUnchecked(size - 1, point);
    }

    private static final MethodHandle swapElement =
        MhBuilder.oneshot3(
            AggrGraphDataArrayList.class, int.class, int.class,
            (list, a, b) -> {
                return StockpileColumnFieldMh.invokeAllExprs(c -> {
                    MhExpr array = arraysMh.getArrayForField(c, list);
                    return c.arrayLikeMh().swapIfNotEmpty(array, a, b);
                });
            });

    private void swapElementsUnchecked(int a, int b) {
        try {
            swapElement.invokeExact(this, a, b);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle copyElement =
        MhBuilder.oneshot3(
            AggrGraphDataArrayList.class, int.class, int.class,
            (list, dst, src) -> {
                return StockpileColumnFieldMh.invokeAllExprs(c -> {
                    MhExpr array = arraysMh.getArrayForField(c, list);
                    return c.arrayLikeMh().copyElementWithinArrayIfNotEmpty(array, dst, src);
                });
            });

    private void copyElementUnchecked(int dst, int src) {
        try {
            copyElement.invokeExact(this, dst, src);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public void copyElement(int dst, int src) {
        checkIndex(dst);
        checkIndex(src);

        if (dst == src) {
            return;
        }

        copyElementUnchecked(dst, src);
    }

    private static final MethodHandle moveElements =
        MhBuilder.oneshot4(
            arraysMh.arraysClass, int.class, int.class, int.class,
            (list, src, dst, count) -> {
                return StockpileColumnFieldMh.invokeAllExprs(c -> {
                    MhExpr array = arraysMh.getArrayForField(c, list);
                    return c.arrayLikeMh().arrayCopyWithinArrayIfNotEmpty(array, src, dst, count);
                });
            });

    private void moveElementsUnchecked(int src, int dst, int count) {
        try {
            moveElements.invokeExact(this, src, dst, count);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private class FullSortAccessorImpl implements ArraySortAccessor {
        @Override
        public boolean less(int i, int j) {
            return tss[i] < tss[j];
        }

        private FullSortAccessorImpl() {
        }

        @Override
        public void swap(int i, int j) {
            swapElementsUnchecked(i, j);
        }

        @Override
        public void move(int from, int to) {
            copyElementUnchecked(to, from);
        }

        @Override
        public void move(int from, int to, int size) {
            moveElementsUnchecked(from, to, size);
        }
    }

    public void sortByTs() {
        if (isEmpty()) {
            return;
        }

        // rule out common case
        if (isSorted()) {
            return;
        }

        FullSortAccessorImpl sortAccessor = new FullSortAccessorImpl();
        // TODO: buffer
        // TODO: sort indices
        StableSortImpl.mergeSortWithoutBuffer(sortAccessor, 0, size);
    }

    private static final MethodHandle mergeTwoIndexesMh =
        MhBuilder.oneshot3(
            arraysMh.arraysClass, int.class, int.class,
            (arrays, readPos, writePos) -> {
                return StockpileMergeMh.merge(
                    arraysMh.getValues(arrays, writePos),
                    arraysMh.getValues(arrays, readPos),
                    c -> arraysMh.hasColumn(arrays, c),
                    arraysMh.setValuesTarget(arrays, writePos));
            });

    private void mergeTwoIndexes(int readPos, int writePos) {
        try {
            mergeTwoIndexesMh.invokeExact(this, readPos, writePos);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public void mergeAdjacent() {
        if (size == 0) {
            return;
        }

        int readPos = 1;
        int writePos = 0;
        for (;;) {
            if (readPos == size) {
                size = writePos + 1;
                break;
            }

            // documentation
            if (writePos >= readPos) {
                throw new IllegalStateException("writePos is greater than or equal to readPos: " + writePos + ">=" + readPos);
            }

            if (tss[readPos] == tss[writePos]) {
                mergeTwoIndexes(readPos, writePos);
                ++readPos;
            } else {
                ++writePos;
                copyElementUnchecked(writePos, readPos);
                ++readPos;
            }
        }
    }

    public void retainIf(AggrDataPointPredicate p) {
        if (size == 0) {
            return;
        }

        int readPos = 0;
        int writePos = 0;

        AggrPointData pointData = new AggrPointData();

        for (;;) {
            if (readPos == size) {
                size = writePos;
                break;
            }

            getDataTo(readPos, pointData);
            if (p.testPoint(pointData)) {
                if (writePos != readPos) {
                    copyElementUnchecked(writePos, readPos);
                }
                writePos++;
            }
            ++readPos;
        }
    }

    public void sortAndMerge() {
        sortByTs();
        mergeAdjacent();
    }

    public AggrGraphDataArrayList toSortedMerged() {
        AggrGraphDataArrayList clone = clone();
        clone.sortAndMerge();
        return clone;
    }

    private static final MethodHandle getData = arraysMh.makeGetPointDataTo();

    @Override
    public void getDataTo(int i, AggrPointData target) {
        try {
            getData.invokeExact(this, i, target);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    @Override
    public long getTsMillis(int index) {
        checkIndex(index);

        return tss[index];
    }

    @Override
    public LongArrayView getTimestamps() {
        return new LongArrayView(tss, 0, size);
    }

    @Override
    public int length() {
        return size;
    }

    @Nonnull
    @Override
    public ValueView valueView() {
        return new ValueView(valuesNum, valuesDenom, 0, size);
    }

    private static final MethodHandle getValueDivided = arraysMh.getValueDivided();

    @Override
    public double getValueDivided(int i) {
        try {
            return (double) getValueDivided.invokeExact(this, i);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public double getDouble(int index) {
        checkIndex(index);
        return valuesNum[index];
    }

    public long getLong(int index) {
        checkIndex(index);
        return longValues[index];
    }

    public long getCount(int index) {
        checkIndex(index);
        return counts[index];
    }

    public long getDenom(int index) {
        checkIndex(index);
        if (valuesDenom instanceof Long) {
            return (long) valuesDenom;
        } else {
            long[] array = (long[]) valuesDenom;
            return array[index];
        }
    }

    @Override
    public LogHistogram getLogHistogram(int index) {
        return logHistograms[index];
    }

    @Override
    public Histogram getHistogram(int index) {
        checkIndex(index);
        return histograms[index];
    }

    @Override
    public SummaryInt64Snapshot getSummaryInt64(int index) {
        checkIndex(index);
        return summariesInt64[index];
    }

    @Override
    public SummaryDoubleSnapshot getSummaryDouble(int index) {
        checkIndex(index);
        return summariesDouble[index];
    }

    public boolean getMergeOrDefault(int index) {
        checkIndex(index);

        return MergeColumn.getOptional(merges, index);
    }

    private static final MethodHandle getValueNumOrDefault =
        arraysMh.getValueOrDefaultForColumn(StockpileColumnField.VALUE_NUM);
    private static final MethodHandle getValueNumOrThrow =
        arraysMh.getValueForColumn(StockpileColumnField.VALUE_NUM);
    private static final MethodHandle setValueNum =
        arraysMh.setValueForField(StockpileColumnField.VALUE_NUM);

    private static final MethodHandle getValueDenomOrDefault =
        arraysMh.getValueOrDefaultForColumn(StockpileColumnField.VALUE_DENOM);
    private static final MethodHandle getValueDenomOrThrow =
        arraysMh.getValueForColumn(StockpileColumnField.VALUE_DENOM);
    private static final MethodHandle setValueDenom =
        arraysMh.setValueForField(StockpileColumnField.VALUE_DENOM);

    private static final MethodHandle getValueObjectOrThrow =
        arraysMh.getValueObjectOrThrow();


    private double getValueNumOrDefault(int index) {
        checkIndex(index);

        try {
            return (double) getValueNumOrDefault.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private double getValueNumOrThrow(int index) {
        checkIndex(index);

        try {
            return (double) getValueNumOrThrow.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private long getValueDenomOrDefault(int index) {
        checkIndex(index);

        try {
            return (long) getValueDenomOrDefault.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private long getValueDenomOrThrow(int index) {
        checkIndex(index);

        try {
            return (long) getValueDenomOrThrow.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }


    @Nonnull
    private ValueObject getValueObjectOrThrow(int index) {
        checkIndex(index);

        try {
            return (ValueObject) getValueObjectOrThrow.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }


    public double getValueDividedOrThrow(int index) {
        return ValueColumn.divide(getValueNumOrThrow(index), getValueDenomOrThrow(index));
    }

    public double getValueDividedOrDefault(int index) {
        return ValueColumn.divide(getValueNumOrDefault(index), getValueDenomOrDefault(index));
    }

    public void setAllStep(long step) {
        if (step == StepColumn.DEFAULT_VALUE) {
            this.stepMillis = Cf.LongArray.emptyArray();
        } else {
            this.stepMillis = step;
        }
    }

    public long[] tss() {
        return Arrays.copyOf(tss, size);
    }

    public double[] values() {
        return Arrays.copyOf(valuesNum, size);
    }

    public void setValues(double[] valuesNum) {
        if (!hasColumn(StockpileColumn.VALUE)) {
            throw new IllegalArgumentException("value column is required");
        }

        if (valuesNum.length != this.getRecordCount()) {
            throw new IllegalArgumentException("different values lengths: " + valuesNum.length + "!=" + this.getRecordCount());
        }

        System.arraycopy(valuesNum, 0, this.valuesNum, 0, this.getRecordCount());
        this.valuesDenom = ValueColumn.DEFAULT_DENOM;
    }

    public void setTsMillis(int index, long tsMillis) {
        checkIndex(index);
        tss[index] = tsMillis;
    }

    public void setValue(int index, ValueObject valueObject) {
        checkIndex(index);
        try {
            setValueNum.invokeExact(this, index, valueObject.num);
            setValueDenom.invokeExact(this, index, valueObject.denom);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public boolean[] merges() {
        return MergeColumn.copyOf(merges, size);
    }

    public long[] counts() {
        return Arrays.copyOf(counts, size);
    }

    public long[] stepMillis() {
        return StepColumn.copyOf(stepMillis, size);
    }

    private static final MethodHandle view = arraysMh.view();

    @Override
    @Nonnull
    public AggrGraphDataArrayListView view() {
        try {
            return (AggrGraphDataArrayListView) view.invokeExact(this);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    @Override
    public int hashCode() {
        return view().hashCode();
    }

    @Override
    public boolean equals(@Nullable Object o) {
        if (this == o) return true;

        return this.view().equals(o);
    }


    public void filterInPlace(AggrDataPointPredicate p) {
        int pos = 0;
        while (pos < size) {
            // TODO: boxing is not needed
            if (!p.testPoint(getAnyPoint(pos))) {
                break;
            }
            ++pos;
        }

        int writePos = pos;
        int readPos = pos + 1;

        while (readPos < size) {
            if (p.testPoint(getAnyPoint(readPos))) {
                copyElementUnchecked(writePos, readPos);
                ++readPos;
                ++writePos;
            } else {
                ++readPos;
            }
        }

        size = writePos;
    }

}
