package ru.yandex.solomon.model.timeseries;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.misc.algo.sort.IsSortedImpl;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.column.ValueView;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.util.collection.array.LongArrayView;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public abstract class AggrGraphDataArrayListOrView implements AggrGraphDataIterable, MemMeasurable {

    public boolean isSorted() {
        return IsSortedImpl.isSorted(getRecordCount(), (i, j) -> getTsMillis(i) < getTsMillis(j));
    }

    public abstract void getDataTo(int i, AggrPointData target);

    public abstract long getTsMillis(int index);

    public abstract LongArrayView getTimestamps();

    public abstract int length();

    @Override
    public int getRecordCount() {
        return length();
    }

    @Override
    public int elapsedBytes() {
        return length() * 4;
    }

    protected void checkIndex(int i) {
        if (i < 0 || i >= length()) {
            throw new IllegalStateException("incorrect index: " + i);
        }
    }

    public long lastTsMillis() {
        if (isEmpty()) {
            throw new IllegalStateException("cannot get last ts millis in empty list");
        }
        return getTsMillis(getRecordCount() - 1);
    }

    @Nonnull
    public abstract ValueView valueView();

    public abstract double getValueDivided(int i);

    public abstract LogHistogram getLogHistogram(int index);

    public abstract Histogram getHistogram(int index);

    public abstract SummaryInt64Snapshot getSummaryInt64(int index);

    public abstract SummaryDoubleSnapshot getSummaryDouble(int index);

    public double[] dividedValues() {
        return valueView().dividedValues();
    }

    public AggrGraphDataArrayListView slice(int from, int to) {
        return view().slice(from, to);
    }

    public abstract AggrGraphDataArrayListView view();

    @Override
    public String toString() {
        var point = RecyclableAggrPoint.newInstance();
        try {
            StringBuilder sb = new StringBuilder();
            sb.append(this.getClass().getSimpleName())
                .append("{")
                .append("columns: "+ columnSet())
                .append(", records: "+ DataSize.shortString(getRecordCount()))
                .append(", points: [");

            point.columnSet = columnSetMask();
            if (getRecordCount() <= 10) {
                int cnt = 0;
                var it = iterator();
                while (it.next(point)) {
                    if (cnt != 0) {
                        sb.append(", ");
                    }
                    sb.append(point);
                    cnt++;
                }
            } else {
                int from = getRecordCount() - 5;
                int cnt = 0;
                while (cnt < getRecordCount()) {
                    if (cnt <= 5 || cnt >= from) {
                        if (cnt != 0) {
                            sb.append(", ");
                        }
                        if (cnt != 5) {
                            getDataTo(cnt, point);
                            sb.append(point);
                        } else {
                            sb.append("...");
                            cnt = from;
                            continue;
                        }
                    }
                    cnt++;
                }
            }

            sb.append("]");
            return sb.append("}").toString();
        } finally {
            point.recycle();
        }
    }

    @Override
    @Nonnull
    public AggrGraphDataArrayListViewIterator iterator() {
        return view().iterator();
    }

    @Nonnull
    public GraphDataArrayList toGraphDataArrayList(MetricType dataType) {
        if (isEmpty()) {
            return new GraphDataArrayList();
        }
        AggrGraphDataListIterator converted = MetricTypeTransfers.of(dataType, MetricType.DGAUGE, iterator());

        double[] values = new double[length()];
        long[] tss = new long[length()];
        int i = 0;
        var point = RecyclableAggrPoint.newInstance();
        while (converted.next(point)) {
            tss[i] = point.getTsMillis();
            values[i] = point.getValueDivided();
            i++;
        }
        point.recycle();

        return new GraphDataArrayList(
                tss,
                values,
                i);
    }

    @Nonnull
    @Deprecated
    @VisibleForTesting
    public GraphData toGraphDataShort() {
        return toGraphDataShort(StockpileColumns.typeByMask(columnSetMask()));
    }

    @Nonnull
    public GraphData toGraphDataShort(MetricType dataType) {
        return toGraphDataArrayList(dataType).buildGraphData();
    }

    @Override
    public abstract long memorySizeIncludingSelf();
}
