package ru.yandex.solomon.model.timeseries;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.commune.mh.builder.MhBuilder;
import ru.yandex.commune.mh.builder.MhCall;
import ru.yandex.commune.mh.builder.MhConst;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.commune.mh.builder.MhExprBool;
import ru.yandex.commune.mh.builder.MhExprEq;
import ru.yandex.commune.mh.builder.MhExprHashCode;
import ru.yandex.commune.mh.builder.MhExprInts;
import ru.yandex.commune.mh.builder.MhIfThenElse;
import ru.yandex.misc.algo.BinarySearch;
import ru.yandex.misc.lang.Validate;
import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.DataPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.StockpileColumnField;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.point.column.ValueObject;
import ru.yandex.solomon.model.point.column.ValueView;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.util.collection.array.LongArrayView;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class AggrGraphDataArrayListView extends AggrGraphDataArrayListOrView {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(AggrGraphDataArrayList.class);

    private static final AggrGraphDataArraysMh<AggrGraphDataArrayListView> arraysMh =
        new AggrGraphDataArraysMh<AggrGraphDataArrayListView>(AggrGraphDataArrayListView.class, MethodHandles.lookup()) {
            @Nonnull
            @Override
            MhExpr capacity(MhExpr arrays) {
                throw new RuntimeException("unreachable");
            }

            @Override
            MhExpr arraysFrom(MhExpr arrays) {
                return MhCall.getInstanceField(arrays, "from");
            }

            @Override
            MhExpr arraysTo(MhExpr arrays) {
                return MhCall.getInstanceField(arrays, "to");
            }

            @Override
            MhExpr indexToArrayIndex(MhExpr arrays, MhExpr index) {
                return MhExprInts.sum(arraysFrom(arrays), index);
            }

            @Override
            MhExpr view(MhExpr list) {
                return list;
            }
        };


    @StockpileColumnField.A(StockpileColumnField.TS)
    @Nonnull
    private final long[] tss;
    @StockpileColumnField.A(StockpileColumnField.VALUE_NUM)
    @Nonnull
    private final double[] sums;
    @StockpileColumnField.A(StockpileColumnField.VALUE_DENOM)
    @Nonnull
    private final Object valuesDenom;
    @StockpileColumnField.A(StockpileColumnField.MERGE)
    @Nonnull
    private final Object merges;
    @StockpileColumnField.A(StockpileColumnField.COUNT)
    @Nonnull
    private final long[] counts;
    @StockpileColumnField.A(StockpileColumnField.STEP)
    @Nonnull
    private final Object stepMillis;
    @Nullable
    @StockpileColumnField.A(StockpileColumnField.LOG_HISTOGRAM)
    private final LogHistogram[] logHistograms;
    @Nullable
    @StockpileColumnField.A(StockpileColumnField.HISTOGRAM)
    private final Histogram[] histograms;
    @Nullable
    @StockpileColumnField.A(StockpileColumnField.ISUMMARY)
    private final SummaryInt64Snapshot[] summariesInt64;
    @Nullable
    @StockpileColumnField.A(StockpileColumnField.DSUMMARY)
    private final SummaryDoubleSnapshot[] summariesDouble;
    @StockpileColumnField.A(StockpileColumnField.LONG_VALUE)
    private final long[] longValues;


    public final int from;
    public final int to;

    public AggrGraphDataArrayListView(
            long[] tss,
            double[] sums,
            Object valuesDenom,
            Object merges,
            long[] counts,
            Object stepMillis,
            LogHistogram[] logHistograms,
            Histogram[] histograms,
            SummaryInt64Snapshot[] summariesInt64,
            SummaryDoubleSnapshot[] summariesDouble,
            long[] longValues,
            int from,
            int to)
    {
        if (from < 0) {
            throw new IllegalArgumentException("\"from\" cannot be < 0");
        }
        if (from > to) {
            throw new IllegalArgumentException("\"from\" cannot be greater than \"to\"");
        }
        if (to > tss.length) {
            throw new IllegalArgumentException("\"to\" cannot be greater than record count");
        }

        this.tss = tss;
        this.sums = sums;
        this.valuesDenom = valuesDenom;
        this.merges = merges;
        this.counts = counts;
        this.stepMillis = stepMillis;
        this.logHistograms = logHistograms;
        this.histograms = histograms;
        this.summariesInt64 = summariesInt64;
        this.summariesDouble = summariesDouble;
        this.longValues = longValues;
        this.from = from;
        this.to = to;
    }

    public static MhExpr newInstance(List<MhExpr> arrays, MhExpr from, MhExpr to) {
        Validate.equals(StockpileColumnField.values().length, arrays.size());

        ArrayList<MhExpr> params = new ArrayList<>();
        params.addAll(arrays);
        params.add(from);
        params.add(to);
        return MhCall.newInstance(AggrGraphDataArrayListView.class, params);
    }

    private static final MethodHandle maskFromArrays =
        arraysMh.maskFromArrays();

    @Nonnull
    public static AggrGraphDataArrayListView empty() {
        return new AggrGraphDataArrayList().view();
    }

    @Override
    public int columnSetMask() {
        try {
            return (int) maskFromArrays.invokeExact(this);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public boolean hasCount() {
        return counts.length != 0;
    }

    @Override
    public int length() {
        return to - from;
    }

    public boolean isEmpty() {
        return length() == 0;
    }

    @Override
    public LongArrayView getTimestamps() {
        return new LongArrayView(tss, from, to);
    }

    private static final MethodHandle hasStepMillis = arraysMh.hasColumn(StockpileColumn.STEP);

    public boolean hasStepMillis() {
        try {
            return (boolean) hasStepMillis.invokeExact(this);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle getData = arraysMh.makeGetPointDataTo();

    @Override
    public void getDataTo(int i, AggrPointData target) {
        try {
            getData.invokeExact(this, i, target);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }


    @Override
    public long getTsMillis(int i) {
        checkIndex(i);
        return tss[indexToArrayIndex(i)];
    }

    @Override
    @Nonnull
    public ValueView valueView() {
        return new ValueView(sums, valuesDenom, from, to);
    }

    private static final MethodHandle getValueDivided = arraysMh.getValueDivided();

    @Override
    public double getValueDivided(int i) {
        try {
            return (double) getValueDivided.invokeExact(this, i);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle getLogHistogram = arraysMh.getValueForColumn(StockpileColumnField.LOG_HISTOGRAM);
    @Override
    public LogHistogram getLogHistogram(int index) {
        try {
            return (LogHistogram) getLogHistogram.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle getHistogram = arraysMh.getValueForColumn(StockpileColumnField.HISTOGRAM);

    @Override
    public Histogram getHistogram(int index) {
        try {
            return (Histogram) getHistogram.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle getSummaryInt64 = arraysMh.getValueForColumn(StockpileColumnField.ISUMMARY);

    @Override
    public SummaryInt64Snapshot getSummaryInt64(int index) {
        try {
            return (SummaryInt64Snapshot) getSummaryInt64.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle getSummaryDouble = arraysMh.getValueForColumn(StockpileColumnField.DSUMMARY);

    @Override
    public SummaryDoubleSnapshot getSummaryDouble(int index) {
        try {
            return (SummaryDoubleSnapshot) getSummaryDouble.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle getValueNum = arraysMh.getValueForColumn(StockpileColumnField.VALUE_NUM);
    private static final MethodHandle getValueDenom = arraysMh.getValueForColumn(StockpileColumnField.VALUE_DENOM);

    public double getValueNum(int i) {
        try {
            return (double) getValueNum.invokeExact(this, i);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public long getValueDenom(int i) {
        try {
            return (long) getValueDenom.invokeExact(this, i);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public ValueObject getValueObject(int i) {
        return new ValueObject(getValueNum(i), getValueDenom(i));
    }

    private static final MethodHandle slice =
        MhBuilder.oneshot3(
            AggrGraphDataArrayListView.class, int.class, int.class,
            (list, from, to) -> {
                List<MhExpr> arrays = arraysMh.getArrays(list);
                return AggrGraphDataArrayListView.newInstance(arrays, from, to);
            });


    @Override
    public AggrGraphDataArrayListView slice(int from, int to) {
        if (from < 0) {
            throw new IllegalArgumentException("Failed to slice: \"from\" cannot be < 0");
        }
        if (to < from) {
            throw new IllegalArgumentException("Failed to slice: \"to\" cannot be less than \"from\"");
        }
        if (to > getRecordCount()) {
            throw new IllegalArgumentException("Failed to slice: \"to\" cannot be greater than record count");
        }

        try {
            return (AggrGraphDataArrayListView) slice.invokeExact(this, this.from + from, this.from + to);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    @Override
    public AggrGraphDataArrayListView view() {
        return this;
    }

    @Nonnull
    public ValueObject avgSum() {
        return valueView().avgSum();
    }


    private double avgSumDerivNum() {
        double sumDerivSum = 0;
        int count = 0;
        for (int i = 1; i < length(); ++i) {
            double d = getValueDivided(i) - getValueDivided(i - 1);
            if (d < 0 || Double.isNaN(d)) {
                continue;
            }
            long dts = getTsMillis(i) - getTsMillis(i - 1);
            if (dts <= 0) {
                throw new IllegalArgumentException("must be sorted");
            }
            sumDerivSum += d / dts;
            count += 1;
        }
        return sumDerivSum * 1000 / count;
    }

    private long avgSumDerivDenom() {
        return ValueColumn.DEFAULT_DENOM;
    }

    public ValueObject avgSumDeriv() {
        return new ValueObject(avgSumDerivNum(), avgSumDerivDenom());
    }

    private static final MethodHandle getStepMillis =
        arraysMh.getValueForSingleFieldColumn(StockpileColumn.STEP);

    private int indexToArrayIndex(int index) {
        checkIndex(index);
        return from + index;
    }

    private long getStepMillis(int index) {
        checkIndex(index);
        try {
            return (long) getStepMillis.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    public long lastStepMillis() {
        return getStepMillis(length() - 1);
    }

    @Nonnull
    public List<DataPoint> getShortPoints() {
        return new AbstractList<DataPoint>() {
            @Override
            public DataPoint get(int index) {
                return new DataPoint(
                    getTsMillis(index),
                    getValueDivided(index));
            }

            @Override
            public int size() {
                return AggrGraphDataArrayListView.this.getRecordCount();
            }
        };
    }

    @Nonnull
    public List<AggrPoint> getFullPoints() {
        return new AbstractList<AggrPoint>() {
            @Override
            public AggrPoint get(int index) {
                return getAggrPoint(index);
            }

            @Override
            public int size() {
                return AggrGraphDataArrayListView.this.getRecordCount();
            }
        };
    }

    private static final MethodHandle getAnyPoint = arraysMh.makeGetAnyPoint();

    @Nonnull
    private AggrPoint getAggrPoint(int index) {
        checkIndex(index);

        try {
            return (AggrPoint) getAnyPoint.invokeExact(this, index);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    @Nonnull
    public AggrGraphDataArrayListView cropSorted(Interval interval, boolean includeOutValues) {
        return cropForResponse(interval.getBeginMillis(), interval.getEndMillis(), includeOutValues);
    }

    @Nonnull
    private AggrGraphDataArrayListView retainGe(long ts, boolean includeOneBefore) {
        if (ts == 0) {
            return this;
        }

        int firstIndex = BinarySearch.firstIndex(0, getRecordCount(), i -> getTsMillis(i) >= ts);
        if (includeOneBefore && firstIndex != 0 && firstIndex != length() && getTsMillis(firstIndex) > ts) {
            --firstIndex;
        }
        return slice(firstIndex, length());
    }

    @Nonnull
    private AggrGraphDataArrayListView retainLe(long ts, boolean includeOneAfter) {
        if (ts == Long.MAX_VALUE) {
            return this;
        }

        int lastIndex = BinarySearch.firstIndex(0, getRecordCount(), i -> getTsMillis(i) > ts);
        if (includeOneAfter && lastIndex != 0 && lastIndex != length() && getTsMillis(lastIndex - 1) < ts) {
            ++lastIndex;
        }
        return slice(0, lastIndex);
    }

    @Nonnull
    public AggrGraphDataArrayListView cropForResponse(long tsFrom, long tsTo, boolean includeOutValues) {
        return this
            .retainGe(tsFrom, includeOutValues)
            .retainLe(tsTo, includeOutValues);
    }

    @Nonnull
    public AggrGraphDataArrayListView deleteBefore(long instantMillis) {
        int firstIndex = BinarySearch.firstIndex(0, getRecordCount(), i -> getTsMillis(i) >= instantMillis);
        return slice(firstIndex, length());
    }

    @Nonnull
    private static MhExpr from(MhExpr view) {
        return MhCall.getInstanceField(view, "from");
    }

    @Nonnull
    private static MhExpr to(MhExpr view) {
        return MhCall.getInstanceField(view, "to");
    }

    private static MhExpr empty(MhExpr view) {
        return MhCall.instanceMethod(view, "isEmpty");
    }

    private static final MethodHandle hashCode =
        MhBuilder.oneshot1(
            AggrGraphDataArrayListView.class,
            view -> {
                MhExpr from = from(view);
                MhExpr to = to(view);
                List<MhExpr> hashCodes = Arrays.stream(StockpileColumnField.values())
                    .map(c -> {
                        MhExpr array = arraysMh.getArrayForField(c, view);
                        MhExpr hashCode = c.arrayLikeMh().hashCodeOfRange(array, from, to);
                        return MhIfThenElse.ifThenElse(
                            c.arrayLikeMh().arrayIsNotEmpty(array),
                            hashCode,
                            MhConst.intConst(0));
                    })
                    .collect(Collectors.toList());
                return MhExprHashCode.hashCode(hashCodes);
            });

    @Override
    public int hashCode() {
        try {
            return (int) hashCode.invokeExact(this);
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    private static final MethodHandle equals =
        MhBuilder.oneshot2(
            AggrGraphDataArrayListView.class, AggrGraphDataArrayListView.class,
            (l1, l2) -> {
                MhExpr from1 = from(l1);
                MhExpr to1 = to(l1);
                MhExpr from2 = from(l2);
                MhExpr to2 = to(l2);

                List<MhExpr> ops = Arrays.stream(StockpileColumnField.values())
                    .map(c -> {
                        // array
                        MhExpr a1 = arraysMh.getArrayForField(c, l1);
                        MhExpr a2 = arraysMh.getArrayForField(c, l2);

                        // array is not empty
                        MhExpr a1n = c.arrayLikeMh().arrayIsNotEmpty(a1);
                        MhExpr a2n = c.arrayLikeMh().arrayIsNotEmpty(a2);

                        return MhIfThenElse.ifThenElse(
                            MhExprEq.eq(a1n, a2n),
                            MhIfThenElse.ifThenElse(
                                a1n, // == a2n
                                c.arrayLikeMh().equalRanges(a1, from1, to1, a2, from2, to2),
                                MhConst.boolConst(true)
                            ),
                            MhConst.boolConst(false));
                    })
                    .collect(Collectors.toList());

                return MhIfThenElse.ifThenElse(
                    MhExprBool.and(empty(l1), empty(l2)),
                    MhConst.boolConst(true),
                    MhExprBool.and(ops)
                );
            });

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof AggrGraphDataArrayListOrView that)) {
            return false;
        }

        try {
            return (boolean) equals.invokeExact(this, that.view());
        } catch (Throwable throwable) {
            throw new RuntimeException(throwable);
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(AggrGraphDataArrayListView.class.getSimpleName());
        sb.append("[");
        for (int i = 0; i < length(); ++i) {
            if (i != 0) {
                sb.append(", ");
            }
            sb.append(getAggrPoint(i));
        }
        sb.append("]");
        return sb.toString();
    }

    @Nonnull
    public AggrGraphDataArrayListView dropPointsByTsBeforeInSorted(long tsMillis) {
        // linear but OK, because used rarely
        for (int i = 0; i < length(); ++i) {
            if (getTsMillis(i) >= tsMillis) {
                return slice(i, length());
            }
        }
        return slice(length(), length());
    }

    @Nonnull
    @Override
    public AggrGraphDataArrayListViewIterator iterator() {
        return new AggrGraphDataArrayListViewIterator(this);
    }

    @Override
    public long memorySizeIncludingSelf() {
        // ListView only references others data, so do not account it
        return SELF_SIZE;
    }
}
