package ru.yandex.solomon.model.timeseries;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.commune.mh.builder.MhBuilder;
import ru.yandex.commune.mh.builder.MhCall;
import ru.yandex.commune.mh.builder.MhConst;
import ru.yandex.commune.mh.builder.MhExpr;
import ru.yandex.commune.mh.builder.MhExprArrays;
import ru.yandex.commune.mh.builder.MhExprInts;
import ru.yandex.commune.mh.builder.MhFieldRef;
import ru.yandex.commune.mh.builder.MhIfThenElse;
import ru.yandex.commune.mh.builder.MhValueTarget;
import ru.yandex.misc.lang.Validate;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointData;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.StockpileColumnField;
import ru.yandex.solomon.model.point.column.StockpileColumnFieldMh;
import ru.yandex.solomon.model.point.column.StockpileColumnsMh;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.point.column.ValueObject;

/**
 * @author Stepan Koltsov
 */
public abstract class AggrGraphDataArraysMh<A extends AggrGraphDataArrayListOrView> {


    final Class<A> arraysClass;
    private final MethodHandles.Lookup lookup;

    public AggrGraphDataArraysMh(Class<A> arraysClass, MethodHandles.Lookup lookup) {
        this.arraysClass = arraysClass;
        this.lookup = lookup;
    }

    Field fieldForColumn(StockpileColumnField column) {
        return column.mh().fieldForColumn(arraysClass);
    }

    @Nonnull
    public MhFieldRef fieldRef(StockpileColumnField column, MhExpr arrays) {
        return new MhFieldRef(arrays, fieldForColumn(column));
    }

    @Nonnull
    MhExpr capacity(MhExpr arrays) {
        Validate.notEquals(arrays, AggrGraphDataArrayListView.class);
        MhExpr tss = fieldRef(StockpileColumn.TS.singleDataField(), arrays).get();
        return MhExprArrays.length(tss);
    }

    @Nonnull
    MhExpr length(MhExpr arrays) {
        return MhCall.instanceMethod(arrays.cast(AggrGraphDataArrayListOrView.class), "length");
    }

    abstract MhExpr arraysFrom(MhExpr arrays);
    abstract MhExpr arraysTo(MhExpr arrays);
    abstract MhExpr indexToArrayIndex(MhExpr arrays, MhExpr index);

    MhExpr view(MhExpr list) {
        List<MhExpr> arrays = getArrays(list);
        MhExpr from = arraysFrom(list);
        MhExpr to = arraysTo(list);
        return AggrGraphDataArrayListView.newInstance(arrays, from, to);
    }

    MethodHandle view() {
        return MhBuilder.oneshot1(arraysClass, this::view);
    }

    MhExpr getArrayForField(StockpileColumnField column, MhExpr arrays) {
        return fieldRef(column, arrays).get();
    }

    Object getArrayForField(StockpileColumnField field, A arrays) {
        return MhBuilder.eval1(arrays, arraysP -> getArrayForField(field, arraysP));
    }

    MhExpr getValueOrDefaultForColumn(StockpileColumnField column, MhExpr arrays, MhExpr index) {
        MhExpr array = getArrayForField(column, arrays);
        return column.arrayLikeMh().getIfNotEmptyOrDefault(array, indexToArrayIndex(arrays, index));
    }

    MethodHandle getValueOrDefaultForColumn(StockpileColumnField field) {
        return MhBuilder.oneshot2(
            arraysClass, int.class,
            (arrays, index) -> getValueOrDefaultForColumn(field, arrays, index));
    }

    MhExpr getValueForColumn(StockpileColumnField column, MhExpr arrays, MhExpr index) {
        MhExpr array = getArrayForField(column, arrays);
        return column.arrayLikeMh().get(array, indexToArrayIndex(arrays, index));
    }

    MethodHandle getValueForColumn(StockpileColumnField column) {
        return MhBuilder.oneshot2(arraysClass, int.class, (a, i) -> getValueForColumn(column, a, i));
    }

    MethodHandle getValueForSingleFieldColumn(StockpileColumn column) {
        return getValueForColumn(column.singleDataField());
    }

    MhExpr getValueObjectOrThrow(MhExpr arrays, MhExpr index) {
        MhExpr num = getValueForColumn(StockpileColumnField.VALUE_NUM, arrays, index);
        MhExpr denom = getValueForColumn(StockpileColumnField.VALUE_DENOM, arrays, index);
        return MhCall.newInstance(ValueObject.class, num, denom);
    }

    MethodHandle getValueObjectOrThrow() {
        return MhBuilder.oneshot2(arraysClass, int.class, this::getValueObjectOrThrow);
    }

    MhExpr setValueForField(StockpileColumnField column, MhExpr list, MhExpr pos, MhExpr value) {
        list.assertType(AggrGraphDataArrayList.class);
        return column.arrayLikeMh().setInField(
            fieldRef(column, list),
            capacity(list),
            pos,
            value);
    }

    MethodHandle setValueForField(StockpileColumnField field) {
        return MhBuilder.oneshot3(
            arraysClass, int.class, field.arrayClass().getComponentType(),
            (list, index, value) -> this.setValueForField(field, list, index, value));
    }

    MhValueTarget setValueForFieldTarget(StockpileColumnField column, MhExpr list, MhExpr pos) {
        return value -> setValueForField(column, list, pos, value);
    }

    List<MhValueTarget> setValuesTarget(MhExpr list, MhExpr pos) {
        list.assertType(AggrGraphDataArrayList.class);
        return Arrays.stream(StockpileColumnField.values())
            .map(d -> setValueForFieldTarget(d, list, pos))
            .collect(Collectors.toList());
    }

    MhExpr getValueDivided(MhExpr list, MhExpr pos) {
        MhExpr num = getValueForColumn(StockpileColumnField.VALUE_NUM, list, pos);
        MhExpr denom = getValueForColumn(StockpileColumnField.VALUE_DENOM, list, pos);
        return ValueColumn.divide(num, denom);
    }

    public MethodHandle getValueDivided() {
        return MhBuilder.oneshot2(arraysClass, int.class, this::getValueDivided);
    }

    // L -> f[]
    MethodHandle getArrayForField(StockpileColumnField column) {
        return MhBuilder.oneshot1(arraysClass, arrays -> getArrayForField(column, arrays));
    }

    MhExpr hasColumn(MhExpr arrays, StockpileColumn column) {
        return hasField(arrays, column.maskField());
    }

    MhExpr hasField(MhExpr arrays, StockpileColumnField field) {
        MhExpr array = getArrayForField(field, arrays);
        return field.arrayLikeMh().arrayIsNotEmpty(array);
    }

    MethodHandle hasColumn(StockpileColumn column) {
        return MhBuilder.oneshot1(arraysClass, a -> hasColumn(a, column));
    }

    List<MhExpr> getArrays(MhExpr arrays) {
        return Arrays.stream(StockpileColumnField.values())
            .map(c -> getArrayForField(c, arrays))
            .collect(CollectorsF.toList());
    }

    private static boolean sameBoolean(boolean a, boolean b) {
        if (a != b) {
            throw new RuntimeException("different booleans: " + a + "!=" + b);
        }
        return a;
    }

    private static MhExpr sameBoolean(MhExpr a, MhExpr b) {
        return MhCall.staticMethod(AggrGraphDataArraysMh.class, "sameBoolean", a, b);
    }

    private MhExpr checkColumnConsistent(MhExpr list, StockpileColumn column) {
        if (column.dataFields().size() <= 1) {
            return MhConst.voidConst();
        }

        List<MhExpr> arrays = getArrays(list);
        List<MhExpr> arraysIsNotEmpty = column.dataFields().stream()
            .map(d -> d.arrayLikeMh().arrayIsNotEmpty(arrays.get(d.ordinal())))
            .collect(Collectors.toList());

        MhExpr r = arraysIsNotEmpty.stream().reduce(AggrGraphDataArraysMh::sameBoolean).get();
        return r.cast(void.class);
    }

    private MhExpr checkConsitent(MhExpr list) {
        return StockpileColumnsMh.invokeAllExprs(c -> checkColumnConsistent(list, c));
    }

    public MethodHandle checkConsistent() {
        return MhBuilder.oneshot1(arraysClass, this::checkConsitent);
    }

    public List<MhExpr> getValuesOrDefault(MhExpr arrays, MhExpr index) {
        return Arrays.stream(StockpileColumnField.values())
            .map(c -> getValueOrDefaultForColumn(c, arrays, index))
            .collect(Collectors.toList());
    }

    public List<MhExpr> getValues(MhExpr arrays, MhExpr index) {
        return Arrays.stream(StockpileColumnField.values())
            .map(c -> getValueForColumn(c, arrays, index))
            .collect(Collectors.toList());
    }

    MhExpr setArrayForField(StockpileColumnField column, MhExpr arrays, MhExpr array) {
        return fieldRef(column, arrays).set(array);
    }

    MhExpr getAnyPoint(MhExpr arrays, MhExpr index) {
        List<MhExpr> arrayExprs = Arrays.stream(StockpileColumnField.values())
            .map(c -> getArrayForField(c, arrays))
            .collect(Collectors.toList());
        return getAggrPointFromArrays(indexToArrayIndex(arrays, index), arrayExprs);
    }

    private static MhExpr getAggrPointFromArrays(MhExpr pos, List<? extends MhExpr> arrays) {
        List<MhExpr> values = Arrays.stream(StockpileColumnField.values())
            .map(d -> {
                MhExpr array = arrays.get(d.ordinal());
                return d.arrayLikeMh().getIfNotEmptyOrDefault(array, pos);
            })
            .collect(Collectors.toList());

        ArrayList<MhExpr> params = new ArrayList<>();
        params.add(AggrGraphDataArraysMh.maskFromArrays(arrays));
        params.addAll(values);
        return MhCall.newInstance(AggrPoint.class, params.toArray(new MhExpr[0]));
    }

    // L, int -> AggrPoint
    MethodHandle makeGetAnyPoint() {
        return MhBuilder.oneshot2(arraysClass, int.class, this::getAnyPoint);
    }

    private MhExpr getAnyPointTo(MhExpr list, MhExpr pos, MhExpr target) {
        return StockpileColumnFieldMh.invokeAllExprs(f -> {
            MhExpr cond = hasField(list, f);
            var value = getValueForColumn(f, list, pos);
            final MhExpr copy;
            if (f.mutable) {
                final MhExpr targetValue = f.mh().columnGetter(target);
                copy = MhCall.staticMethod(f.column.columnClass, "copy", value, targetValue)
                    .assertType(f.arrayType.elementClass());
            } else {
                copy = value;
            }
            MhExpr set = f.mh().columnSetter(target, copy);
            return set.onlyIf(cond);
        });
    }

    MethodHandle makeGetPointDataTo() {
        return MhBuilder.oneshot3(arraysClass, int.class, AggrPointData.class, this::getAnyPointTo);
    }

    @Nonnull
    private static MhExpr maskFromArray(StockpileColumn column, MhExpr arrayLike) {
        return MhIfThenElse.ifThenElse(
            column.maskField().arrayLikeMh().arrayIsNotEmpty(arrayLike),
            MhConst.intConst(column.mask()),
            MhConst.intConst(0));
    }

    // f[]... -> int
    private static MethodHandle makeMaskFromArrays() {
        return MhBuilder.oneshot(StockpileColumnFieldMh.columnArraysOrCompact(), AggrGraphDataArraysMh::maskFromArrays);
    }

    public static MhExpr maskFromArrays(List<? extends MhExpr> arrays) {
        Validate.equals(StockpileColumnField.values().length, arrays.size());

        MhExpr[] masks = Arrays.stream(StockpileColumn.values())
            .map(c -> maskFromArray(c, arrays.get(c.maskField().ordinal())))
            .toArray(MhExpr[]::new);

        return MhExprInts.or(masks).assertType(int.class);
    }

    private MhExpr maskFromArrays(MhExpr arrays) {
        List<MhExpr> arrayExprs = getArrays(arrays);
        return maskFromArrays(arrayExprs);
    }

    // L -> int
    MethodHandle maskFromArrays() {
        return MhBuilder.oneshot1(arraysClass, this::maskFromArrays);
    }

}
