package ru.yandex.solomon.model.timeseries;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DerivAggrGraphDataIterator extends AggrGraphDataListIterator {
    private AggrGraphDataListIterator source;
    private AggrPoint prev = new AggrPoint();
    private long deltaTime = 0;
    private long deltaValue = 0;
    private long lastCount = 0;

    private DerivAggrGraphDataIterator(AggrGraphDataListIterator source) {
        super((source.columnSetMask() & ~StockpileColumn.LONG_VALUE.mask()) | StockpileColumn.VALUE.mask());
        this.source = source;
        source.next(prev);
    }

    @Override
    public int estimatePointsCount() {
        return source.estimatePointsCount();
    }

    public static AggrGraphDataListIterator of(AggrGraphDataIterable source) {
        return of(source.iterator());
    }

    public static AggrGraphDataListIterator of(AggrGraphDataListIterator source) {
        if (source.estimatePointsCount() == 0) {
            return EmptyAggrGraphDataIterator.INSTANCE;
        }

        if (!source.hasColumn(StockpileColumn.LONG_VALUE)) {
            throw new IllegalArgumentException("Deriv able apply only on longValue column, but column set was: " + source.columnSet());
        }

        return new DerivAggrGraphDataIterator(source);
    }

    @Override
    public boolean next(AggrPoint target) {
        while (source.next(target)) {
            if (prev.tsMillis == TsColumn.DEFAULT_VALUE) {
                target.copyTo(prev);
                continue;
            }

            if (isResetRequired(target)) {
                target.copyTo(prev);
                target.columnSet = columnSetMask();
                target.valueNum = deltaValue;
                target.valueDenom = deltaTime;
                target.count = lastCount;
                return true;
            }

            deltaValue = target.longValue - prev.longValue;
            deltaTime = target.getTsMillis() - prev.getTsMillis();
            lastCount = target.count;
            target.copyTo(prev);
            target.columnSet = columnSetMask();
            target.valueNum = deltaValue;
            target.valueDenom = deltaTime;
            ValueColumn.validateOrThrow(deltaValue, deltaTime);
            return true;
        }

        return false;
    }

    private boolean isResetRequired(AggrPoint target) {
        if (Long.compareUnsigned(target.longValue, prev.longValue) < 0) {
            return true;
        }

        if (prev.count != target.count) {
            return true;
        }

        if (!ValueColumn.isValidDenom(target.getTsMillis() - prev.getTsMillis())) {
            return true;
        }

        return hasColumn(StockpileColumn.STEP)
                && prev.stepMillis != StepColumn.DEFAULT_VALUE
                && (target.tsMillis - prev.tsMillis) >  prev.stepMillis;
    }
}
