package ru.yandex.solomon.model.timeseries;

import java.util.AbstractList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.DoubleUnaryOperator;
import java.util.function.LongPredicate;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.algo.BinarySearch;
import ru.yandex.solomon.model.point.DataPoint;
import ru.yandex.solomon.model.point.column.ValueView;
import ru.yandex.solomon.model.point.predicate.DataPointPredicate;
import ru.yandex.solomon.util.collection.array.DoubleArrayView;
import ru.yandex.solomon.util.collection.array.LongArrayView;

/**
* @author Stepan Koltsov
*/
@ParametersAreNonnullByDefault
public class GraphData implements GraphDataList {
    public static final GraphData empty = new GraphData();

    /** Millis, sorted, no dups */
    @Nonnull
    private final long[] timestampsArray;
    /** NaN means gap */
    @Nonnull
    private final double[] valuesArray;
    private final int from;
    private final int to;

    private static int sameLength(long[] timestamps, double[] values) {
        if (timestamps.length != values.length) {
            throw new IllegalArgumentException("different timestamp arrays lengths: " + timestamps.length + "!=" + values.length);
        }
        return timestamps.length;
    }

    public GraphData(long[] timestamps, double[] values, SortedOrCheck sortedOrCheck) {
        this(timestamps, values, sameLength(timestamps, values), sortedOrCheck);
    }

    public GraphData(long[] timestamps, double[] values, int length, SortedOrCheck sortedOrCheck) {
        if (length > timestamps.length) {
            throw new IllegalArgumentException("different length of timestamp arrays: " + length + "!=" + timestamps.length);
        }
        if (length > values.length) {
            throw new IllegalArgumentException("different lengths of value arrays: " + length + "!=" + values.length);
        }

        this.timestampsArray = timestamps;
        this.valuesArray = values;
        this.from = 0;
        this.to = length;

        if (sortedOrCheck == SortedOrCheck.CHECK) {
            checkSorted();
        }
    }

    private GraphData(long[] timestampsArray, double[] valuesArray, int from, int to, SortedOrCheck sortedOrCheck) {
        // validate
        new LongArrayView(timestampsArray, from, to);
        new DoubleArrayView(valuesArray, from, to);

        this.timestampsArray = timestampsArray;
        this.valuesArray = valuesArray;
        this.from = from;
        this.to = to;
    }

    public ValueView getValuesView() {
        return new ValueView(valuesArray, from, to);
    }

    private void checkSorted() {
        new Timeline(getTimestamps(), SortedOrCheck.CHECK);
    }

    public static GraphData graphData(long tsMillis, double value) {
        return new GraphData(new long[]{ tsMillis }, new double[] { value }, SortedOrCheck.CHECK);
    }

    public static GraphData graphData(long tsMillis1, double value1, long tsMillis2, double value2) {
        return new GraphData(new long[]{ tsMillis1, tsMillis2 }, new double[] { value1, value2 }, SortedOrCheck.CHECK);
    }

    public static GraphData graphData(long tsMillis1, double value1, long tsMillis2, double value2, long tsMillis3, double value3) {
        return new GraphData(new long[]{ tsMillis1, tsMillis2, tsMillis3 }, new double[] { value1, value2, value3 }, SortedOrCheck.CHECK);
    }

    public int length() {
        return to - from;
    }

    public boolean isEmpty() {
        return length() == 0;
    }

    @Override
    public void visit(PointConsumer consumer) {
        LongArrayView tsView = getTimestamps();
        DoubleArrayView doubleView = getValues();
        for (int i = 0; i < length(); ++i) {
            consumer.consume(tsView.at(i), doubleView.at(i));
        }
    }

    public enum SecondsMarker {
        M
    }

    public GraphData(long[] seconds, double[] values, SecondsMarker M, SortedOrCheck sorted) {
        this(TimelineUtils.secondsToMillis(seconds), values, sorted);
    }

    public GraphData(Timeline timeline, double[] values) {
        this(timeline.getPointsMillis().copyOrArray(), values, SortedOrCheck.SORTED_UNIQUE);
    }

    /**
     * @param dataPoints must be sorted
     */
    public GraphData(List<DataPoint> dataPoints) {
        int size = dataPoints.size();
        long[] timestamps = new long[size];
        double[] values = new double[size];
        for (int i = 0; i < size; i++) {
            DataPoint dataPoint = dataPoints.get(i);
            timestamps[i] = dataPoint.getTsMillis();
            values[i] = dataPoint.getValue();
        }
        this.timestampsArray = timestamps;
        this.valuesArray = values;
        this.from = 0;
        this.to = size;

        checkSorted();
    }

    private GraphData() {
        this(Cf.LongArray.emptyArray(), Cf.DoubleArray.emptyArray(), SortedOrCheck.SORTED_UNIQUE);
    }

    public GraphData(SortedMap<Long, Double> millisToValues) {
        int size = millisToValues.size();
        long[] timestamps = new long[size];
        double[] values = new double[size];
        int i = 0;
        for (Map.Entry<Long, Double> entry : millisToValues.entrySet()) {
            timestamps[i] = entry.getKey();
            values[i] = entry.getValue();
            ++i;
        }
        if (i != size) {
            throw new IllegalStateException("unreachable");
        }
        this.timestampsArray = timestamps;
        this.valuesArray = values;
        this.from = 0;
        this.to = size;
    }

    @Nonnull
    public LongArrayView getTimestamps() {
        return new LongArrayView(timestampsArray, from, to);
    }

    @Nonnull
    public DoubleArrayView getValues() {
        return new DoubleArrayView(valuesArray, from, to);
    }

    public Timeline getTimeline() {
        return new Timeline(getTimestamps(), SortedOrCheck.SORTED_UNIQUE);
    }

    public boolean isGapAtLeft(int index) {
        return index == 0 || Double.isNaN(getValues().at(index - 1));
    }

    public boolean isGapAtRight(int index) {
        return index == length() - 1 || Double.isNaN(getValues().at(index + 1));
    }

    @Nonnull
    public List<DataPoint> getShortPoints() {
        return new AbstractList<DataPoint>() {
            @Override
            public DataPoint get(int index) {
                return new DataPoint(getTimestamps().at(index), getValues().at(index));
            }

            @Override
            public int size() {
                return length();
            }
        };
    }

    @Nonnull
    public DataPoint single() {
        if (length() != 1) {
            throw new IllegalStateException("not a single point: " + length());
        }
        return new DataPoint(getTimestamps().at(0), getValues().at(0));
    }

    @Nonnull
    public GraphData mapValues(DoubleUnaryOperator f) {
        if (isEmpty()) {
            return this;
        }

        double[] values = getValues().stream().map(f).toArray();
        return new GraphData(getTimestamps().copyOrArray(), values, SortedOrCheck.SORTED_UNIQUE);
    }

    private boolean hasNaNs() {
        return getValues().stream().anyMatch(Double::isNaN);
    }

    @Nonnull
    public GraphData dropConsecutiveNaNs() {
        if (isEmpty() || !hasNaNs()) {
            return this;
        }

        long[] timestamps = new long[this.length()];
        double[] values = new double[this.length()];

        int firstNonNan = 0;
        while (firstNonNan < this.length() && Double.isNaN(this.getValues().at(firstNonNan))) {
            ++firstNonNan;
        }

        int output = 0;
        for (int i = firstNonNan; i < this.length(); i++) {
            long timestamp = this.getTimestamps().at(i);
            double value = this.getValues().at(i);

            if (Double.isNaN(value)) {
                if (i == this.length() - 1) {
                    continue;
                }

                if (Double.isNaN(this.getValues().at(i + 1))) {
                    continue;
                }
            }

            timestamps[output] = timestamp;
            values[output] = value;
            ++output;
        }

        return new GraphData(timestamps, values, output, SortedOrCheck.SORTED_UNIQUE);
    }

    @Nonnull
    public GraphData replaceNaNWithZero() {
        if (!hasNaNs()) {
            return this;
        }

        return mapValues(d -> !Double.isNaN(d) ? d : 0);
    }

    private static double merge(double a, double b) {
        if (!Double.isNaN(b)) {
            return b;
        } else {
            return a;
        }
    }

    public static GraphData of(DataPoint... points) {
        return new GraphData(Arrays.asList(points));
    }

    public static GraphData merge(GraphData a, GraphData b) {
        if (a.isEmpty()) {
            return b;
        }
        if (b.isEmpty()) {
            return a;
        }
        GraphDataArrayList r = new GraphDataArrayList(a.length() + b.length());
        int ai = 0;
        int bi = 0;

        while (ai < a.length() && bi < b.length()) {
            long ats = a.getTimestamps().at(ai);
            long bts = b.getTimestamps().at(bi);
            double av = a.getValues().at(ai);
            double bv = b.getValues().at(bi);
            if (ats == bts) {
                r.add(ats, merge(av, bv));
                ++ai;
                ++bi;
            } else if (ats < bts) {
                r.add(ats, av);
                ++ai;
            } else {
                r.add(bts, bv);
                ++bi;
            }
        }

        while (ai < a.length()) {
            r.add(a.getTimestamps().at(ai), a.getValues().at(ai));
            ++ai;
        }
        while (bi < b.length()) {
            r.add(b.getTimestamps().at(bi), b.getValues().at(bi));
            ++bi;
        }

        return r.buildGraphDataFromSortedUniqueNoCheck();
    }

    @Nonnull
    public GraphData slice(int from, int to) {
        if (from < 0 || from > to || to > length()) {
            throw new IllegalArgumentException("from " + from + " to " + to + ", source length " + length());
        }

        return new GraphData(
            timestampsArray,
            valuesArray,
            this.from + from,
            this.from + to,
            SortedOrCheck.SORTED_UNIQUE);
    }

    @Nonnull
    private GraphData drop(int count) {
        if (count == 0) {
            return this;
        } else if (count >= length()) {
            return empty;
        } else {
            return slice(count, length());
        }
    }

    @Nonnull
    private GraphData take(int count) {
        if (count == 0) {
            return empty;
        } else if (count >= length()) {
            return this;
        } else {
            return slice(0, count);
        }
    }

    @Nonnull
    public GraphData dropLt(long instant) {
        if (isEmpty()) {
            return this;
        }
        LongArrayView timestamps = getTimestamps();
        if (timestamps.first() >= instant) {
            return this;
        }
        if (timestamps.last() < instant) {
            return empty;
        }
        int pos = BinarySearch.firstIndex(timestamps.length(), i -> timestamps.at(i) >= instant);
        return drop(pos);
    }

    @Nonnull
    public GraphData dropLe(long instant) {
        if (isEmpty()) {
            return this;
        }
        LongArrayView timestamps = getTimestamps();
        if (timestamps.first() > instant) {
            return this;
        }
        if (timestamps.last() <= instant) {
            return empty;
        }
        int pos = BinarySearch.firstIndex(timestamps.length(), i -> timestamps.at(i) > instant);
        return drop(pos);
    }

    @Nonnull
    public GraphData dropGt(long instant) {
        if (isEmpty()) {
            return this;
        }
        LongArrayView timestamps = getTimestamps();
        if (timestamps.last() <= instant) {
            return this;
        }
        if (timestamps.first() > instant) {
            return empty;
        }
        int pos = BinarySearch.firstIndex(timestamps.length(), i -> timestamps.at(i) > instant);
        return take(pos);
    }

    @Nonnull
    public GraphData dropGe(long instant) {
        if (isEmpty()) {
            return this;
        }
        LongArrayView timestamps = getTimestamps();
        if (timestamps.last() < instant) {
            return this;
        }
        if (timestamps.first() >= instant) {
            return empty;
        }
        int pos = BinarySearch.firstIndex(timestamps.length(), i -> timestamps.at(i) >= instant);
        return take(pos);
    }

    @Nonnull
    public GraphData deriv() {
        if (this.length() < 2) {
            return GraphData.empty;
        }
        long[] timestamps = this.getTimestamps().drop(1).copyToArray();
        double[] values = new double[timestamps.length];
        for (int i = 0; i < timestamps.length; ++i) {
            double d = (this.getValues().at(i + 1) - this.getValues().at(i)) * 1000 / (this.getTimestamps().at(i + 1) - this.getTimestamps().at(i));
            // negative is not possible in deriv
            values[i] = d >= 0 ? d : Double.NaN;
        }
        return new GraphData(timestamps, values, SortedOrCheck.SORTED_UNIQUE);
    }

    public GraphData derivWithNegative() {
        long[] timestamps = this.getTimestamps().drop(1).copyToArray();
        double[] values = new double[timestamps.length];
        for (int i = 0; i < timestamps.length; ++i) {
            double dx = (this.getValues().at(i + 1) - this.getValues().at(i)) * 1000;
            long dy = this.getTimestamps().at(i + 1) - this.getTimestamps().at(i);
            double d = dx / dy;
            values[i] = d;
        }

        return new GraphData(timestamps, values, SortedOrCheck.SORTED_UNIQUE);
    }

    public GraphData deltaToRate() {
        if (this.length() < 2) {
            return GraphData.empty;
        }
        long[] timestamps = this.getTimestamps().copyToArray();
        double[] values = new double[timestamps.length];
        // hack for first point
        values[0] = 1000 * getValues().at(0) / (getTimestamps().at(1) - getTimestamps().at(0));
        for (int i = 1; i < timestamps.length; ++i) {
            double dx = getValues().at(i);
            long dy = getTimestamps().at(i) - getTimestamps().at(i - 1);
            values[i] = 1000 * dx / dy;
        }
        return new GraphData(timestamps, values, SortedOrCheck.SORTED_UNIQUE);
    }

    public GraphData rateToDelta() {
        if (this.length() < 2) {
            return GraphData.empty;
        }
        long[] timestamps = this.getTimestamps().copyToArray();
        double[] values = new double[timestamps.length];
        // hack for first point
        values[0] =  getValues().at(0) * (getTimestamps().at(1) - getTimestamps().at(0)) / 1000;
        for (int i = 1; i < timestamps.length; ++i) {
            double dx = getValues().at(i);
            long dy = getTimestamps().at(i) - getTimestamps().at(i - 1);
            values[i] =  dx * dy / 1000;
        }
        return new GraphData(timestamps, values, SortedOrCheck.SORTED_UNIQUE);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        GraphData that = (GraphData) o;

        if (length() != that.length()) {
            return false;
        }

        for (int i = 0; i < getTimestamps().length(); i++) {
            if (Double.compare(getValues().at(i), that.getValues().at(i)) != 0) {
                return false;
            }
            if ((getTimestamps().at(i) != that.getTimestamps().at(i)) && (!Double.isNaN(getValues().at(i)))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode() {
        int result = getTimestamps().hashCode();
        result = 31 * result + getValues().hashCode();
        return result;
    }

    @Override
    public String toString() {
        return new GraphDataArrayList(getTimestamps().copyOrArray(), getValues().copyOrArray()).toString();
    }

    public TreeMap<Long, Double> toTreeMap() {
        TreeMap<Long, Double> r = new TreeMap<>();
        for (int i = 0; i < getTimestamps().length(); ++i) {
            r.put(getTimestamps().at(i), getValues().at(i));
        }
        return r;
    }

    @Nonnull
    public GraphDataArrayList toArrayList() {
        return new GraphDataArrayList(getTimestamps().copyToArray(), getValues().copyToArray());
    }

    public AggrGraphDataArrayList toAggrGraphDataArrayList() {
        return AggrGraphDataArrayList.of(new GraphDataAsAggrIterable(this));
    }

    @Nonnull
    public GraphData filter(DataPointPredicate p) {
        GraphDataArrayList arrayList = toArrayList();
        arrayList.retainIf(p);
        return arrayList.buildGraphDataFromSortedUniqueNoCheck();
    }

    @Nonnull
    public GraphData filterNonNan() {
        return filter((tsMillis, value) -> !Double.isNaN(value));
    }

    @Nonnull
    public GraphData filterByTs(LongPredicate tsPredicate) {
        return filter(((tsMillis, value) -> tsPredicate.test(tsMillis)));
    }

    @Nonnull
    public GraphData sparseNaNs() {
        if (length() <= 2) {
            return this;
        }
        double[] values = new double[length()];
        long[] ts = new long[length()];

        values[0] = this.getValues().at(0);
        values[length() - 1] = this.getValues().at(length() - 1);
        ts[0] = this.getTimestamps().at(0);
        ts[length() - 1] = this.getTimestamps().at(length() - 1);

        for (int i = 1; i < values.length - 1; i++) {
            values[i] = this.getValues().at(i);
            if (Double.isNaN(values[i])) {
                ts[i] = (this.getTimestamps().at(i + 1) + this.getTimestamps().at(i - 1)) / 2;
            } else {
                ts[i] = this.getTimestamps().at(i);
            }
        }

        return new GraphData(ts, values, SortedOrCheck.SORTED_UNIQUE);
    }
}
