package ru.yandex.solomon.model.timeseries;

import java.util.Arrays;
import java.util.stream.Collectors;
import java.util.stream.LongStream;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.algo.arrayList.ArrayListImpl;
import ru.yandex.misc.algo.sort.ArraySortAccessor;
import ru.yandex.misc.algo.sort.StableSortImpl;
import ru.yandex.solomon.model.point.column.StockpileColumnField;
import ru.yandex.solomon.model.point.predicate.DataPointPredicate;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Stepan Koltsov
 *
 * @see AggrGraphDataArrayList
 */
@ParametersAreNonnullByDefault
public class GraphDataArrayList implements GraphDataList, Cloneable {
    // unsorted
    @Nonnull
    @StockpileColumnField.A(StockpileColumnField.TS)
    private long[] timestamps;
    @Nonnull
    @StockpileColumnField.A(StockpileColumnField.VALUE_NUM)
    private double[] values;
    private int size = 0;

    public GraphDataArrayList() {
        this.timestamps = Cf.LongArray.emptyArray();
        this.values = Cf.DoubleArray.emptyArray();
    }

    public GraphDataArrayList(int capacity) {
        timestamps = new long[capacity];
        values = new double[capacity];
    }

    public GraphDataArrayList(long[] timestamps, double[] values) {
        this.timestamps = timestamps;
        this.values = values;

        if (timestamps.length != values.length) {
            throw new IllegalArgumentException("different lengths: " + timestamps.length + "!=" + values.length);
        }

        this.size = timestamps.length;
    }

    public GraphDataArrayList(long[] timestamps, double[] values, int size) {
        this.timestamps = timestamps;
        this.values = values;

        if (timestamps.length != values.length) {
            throw new IllegalArgumentException("different lengths: " + timestamps.length + "!=" + values.length);
        }

        if (size > timestamps.length) {
            throw new IllegalArgumentException("size is greater than timestamp array length");
        }

        this.size = size;
    }

    public static GraphDataArrayList of(long ts, double value) {
        return new GraphDataArrayList(new long[] { ts }, new double[] { value });
    }

    public static GraphDataArrayList of(long ts1, double value1, long ts2, double value2) {
        return new GraphDataArrayList(new long[] { ts1, ts2 }, new double[] { value1, value2 });
    }

    public boolean sorted() {
        for (int i = 0; i < size - 1; ++i) {
            if (timestamps[i + 1] < timestamps[i]) {
                return false;
            }
        }
        return true;
    }

    private class SortAccessor implements ArraySortAccessor {

        @Override
        public boolean less(int i, int j) {
            return Long.compareUnsigned(timestamps[i], timestamps[j]) < 0;
        }

        @Override
        public void swap(int i, int j) {
            Cf.LongArray.swapElements(timestamps, i, j);
            Cf.DoubleArray.swapElements(values, i, j);
        }

        @Override
        public void move(int from, int to) {
            timestamps[to] = timestamps[from];
            values[to] = values[from];
        }

        @Override
        public void move(int from, int to, int size) {
            System.arraycopy(timestamps, from, timestamps, to, size);
            System.arraycopy(values, from, values, to, size);
        }
    }

    public void sortByTs() {
        StableSortImpl.mergeSortWithoutBuffer(new SortAccessor(), 0, size);
    }

    /**
     * @see AggrGraphDataArrayList#mergeAdjacent()
     */
    public void mergeAdjacent() {
        if (size == 0) {
            return;
        }

        int readPos = 1;
        int writePos = 0;
        for (;;) {
            if (readPos == size) {
                size = writePos + 1;
                break;
            }

            if (timestamps[readPos] == timestamps[writePos]) {
                values[writePos] = values[readPos];
                ++readPos;
            } else {
                ++writePos;
                timestamps[writePos] = timestamps[readPos];
                values[writePos] = values[readPos];
                ++readPos;
            }
        }
    }

    public void retainIf(DataPointPredicate p) {
        if (size == 0) {
            return;
        }

        int readPos = 0;
        int writePos = 0;

        for (;;) {
            if (readPos == size) {
                size = writePos;
                break;
            }

            if (p.test(timestamps[readPos], values[readPos])) {
                timestamps[writePos] = timestamps[readPos];
                values[writePos] = values[readPos];
                ++writePos;
            }
            ++readPos;
        }
    }

    @Nonnull
    public long[] getTimestamps() {
        return timestamps;
    }

    @Nonnull
    public double[] getValues() {
        return values;
    }

    private class AccessorImpl implements ArrayListImpl.Accessor {

        @Override
        public int capacity() {
            return timestamps.length;
        }

        @Override
        public int size() {
            return size;
        }

        @Override
        public void reserveExact(int newCapacity) {
            timestamps = Arrays.copyOf(timestamps, newCapacity);
            values = Arrays.copyOf(values, newCapacity);
        }
    }

    public void add(long ts, double value) {
        ArrayListImpl.prepareAdd(new AccessorImpl());
        timestamps[size] = ts;
        values[size] = value;
        ++size;
    }

    public void addAll(GraphDataList value) {
        value.visit(this::add);
    }



    public boolean isEmpty() {
        return size == 0;
    }

    public void visitFrom(int from, PointConsumer consumer) {
        if (from > size) {
            throw new IllegalArgumentException("from is greater than size");
        }
        for (int i = 0; i < size; ++i) {
            consumer.consume(timestamps[i], values[i]);
        }
    }

    @Override
    public void visit(PointConsumer consumer) {
        visitFrom(0, consumer);
    }

    private int lastIndex() {
        if (isEmpty()) {
            throw new IllegalStateException("cannot get last index in empty list");
        }
        return size - 1;
    }

    public long lastTimestamp() {
        return timestamps[lastIndex()];
    }


    public double lastValue() {
        return values[lastIndex()];
    }

    public void replaceLastValue(double value) {
        values[lastIndex()] = value;
    }

    @Override
    protected GraphDataArrayList clone() {
        GraphDataArrayList r;
        try {
            r = (GraphDataArrayList) super.clone();
            r.timestamps = Arrays.copyOf(timestamps, size);
            r.values = Arrays.copyOf(values, size);
            return r;
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Build graph data from unsorted.
     */
    @Nonnull
    public GraphData buildGraphData() {
        return clone().intoGraphData();
    }

    @Nonnull
    private GraphData intoGraphData() {
        sortByTs();
        mergeAdjacent();
        GraphData r = new GraphData(
            Arrays.copyOf(timestamps, size),
            Arrays.copyOf(values, size),
            SortedOrCheck.SORTED_UNIQUE);
        reset();
        return r;
    }

    private void reset() {
        size = 0;
        timestamps = Cf.LongArray.emptyArray();
        values = Cf.DoubleArray.emptyArray();
    }

    @Nonnull
    public GraphData buildGraphDataFromSortedUniqueNoCheck() {
        if (timestamps.length == size) {
            return new GraphData(timestamps, values, SortedOrCheck.SORTED_UNIQUE);
        } else {
            return new GraphData(Arrays.copyOf(timestamps, size), Arrays.copyOf(values, size), SortedOrCheck.SORTED_UNIQUE);
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null || obj.getClass() != this.getClass()) {
            return false;
        }

        GraphDataArrayList that = (GraphDataArrayList) obj;

        if (this.size != that.size) {
            return false;
        }

        for (int i = 0; i < size; ++i) {
            if (this.timestamps[i] != that.timestamps[i]) {
                return false;
            }
            if (this.values[i] != that.values[i]) {
                return false;
            }
        }

        return true;
    }

    @Override
    public int hashCode() {
        int hash = 1;
        for (int i = 0; i < size; ++i) {
            hash = hash * 31 + Long.hashCode(timestamps[i]);
            hash = hash * 31 + Double.hashCode(values[i]);
        }
        return hash;
    }

    private LongStream tsStream() {
        return Arrays.stream(timestamps, 0, size);
    }

    @Override
    public String toString() {
        return "GraphDataArrayList{" +
                "timestamps=" + tsStream().mapToObj(InstantUtils::formatToMillis).collect(Collectors.joining(", ", "[", "]")) +
                ", values=" + Cf.DoubleArray.toStringOfRange(values, 0, size) +
                '}';
    }
}
