package ru.yandex.solomon.model.timeseries;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class IntegrateAggrGraphDataIterator extends AggrGraphDataListIterator {
    private AggrGraphDataListIterator source;
    private AggrPoint next = new AggrPoint();
    private long prevTsMillis;
    private long prevResult;
    private long prevStepMillis;

    private IntegrateAggrGraphDataIterator(int mask, AggrGraphDataListIterator source) {
        super(mask);
        this.source = source;
    }

    public static AggrGraphDataListIterator of(AggrGraphDataIterable source) {
        return of(source.iterator());
    }

    public static AggrGraphDataListIterator of(AggrGraphDataListIterator source) {
        if (source.estimatePointsCount() == 0) {
            return EmptyAggrGraphDataIterator.INSTANCE;
        }

        if (!source.hasColumn(StockpileColumn.VALUE)) {
            throw new IllegalArgumentException("Integrate able apply only on value column, but column set was: " + source.columnSet());
        }

        int mask = (source.columnSetMask() & ~StockpileColumn.VALUE.mask()) | StockpileColumn.LONG_VALUE.mask();
        return new IntegrateAggrGraphDataIterator(mask, source);
    }

    @Override
    public boolean next(AggrPoint target) {
        while (nextPoint(target)) {
            if (isResetRequired(target)) {
                prevTsMillis = 0;
                prevResult = 0;
            }

            long value = prevResult;
            if (ValueColumn.isOne(target.getValueDenom())) {
                long deltaTime = getDeltaTime(target);
                if (deltaTime == 0) {
                    continue;
                }
                value += Math.round(target.getValueNum() * deltaTime / 1000);
            } else {
                // happiest case
                value += Math.round(target.getValueNum());
            }

            target.columnSet = columnSetMask();
            target.longValue = value;
            prevTsMillis = target.tsMillis;
            prevResult = target.longValue;
            prevStepMillis = target.stepMillis;
            return true;
        }

        return false;
    }

    private boolean isResetRequired(AggrPoint target) {
        if (!hasColumn(StockpileColumn.STEP)) {
            return false;
        }

        if (prevStepMillis == StepColumn.DEFAULT_VALUE) {
            return false;
        }

        return (target.tsMillis - prevTsMillis) > prevStepMillis;
    }

    private boolean nextPoint(AggrPoint target) {
        if (next.tsMillis != TsColumn.DEFAULT_VALUE) {
            next.copyTo(target);
            next.tsMillis = TsColumn.DEFAULT_VALUE;
            return true;
        }

        return source.next(target);
    }

    private boolean lookaheadNextPoint() {
        if (next.tsMillis != TsColumn.DEFAULT_VALUE) {
            return true;
        }

        return source.next(next);
    }

    private long getDeltaTime(AggrPoint target) {
        if (prevTsMillis != 0) {
            return target.getTsMillis() - prevTsMillis;
        }

        if (hasColumn(StockpileColumn.STEP)) {
            if (prevStepMillis != StepColumn.DEFAULT_VALUE) {
                return prevStepMillis;
            }

            if (target.stepMillis != StepColumn.DEFAULT_VALUE) {
                return target.stepMillis;
            }
        }

        if (!lookaheadNextPoint()) {
            // only one point available, and we can't integrate
            return 0;
        }

        // assume that delta time the same for first point as before first and second point
        return next.tsMillis - target.tsMillis;
    }
}
