package ru.yandex.solomon.model.timeseries;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.CountColumn;
import ru.yandex.solomon.model.point.column.HistogramColumn;
import ru.yandex.solomon.model.point.column.LogHistogramColumn;
import ru.yandex.solomon.model.point.column.LongValueColumn;
import ru.yandex.solomon.model.point.column.MergeColumn;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.point.column.SummaryDoubleColumn;
import ru.yandex.solomon.model.point.column.SummaryInt64Column;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class MergingAggrGraphDataIterator extends AggrGraphDataListIterator {
    private static MergeFunction MAX_AGGREGATE_FUNCTION = new MaxAggregateMergeFunction();
    private static MergeFunction COMBINE_AGGREGATE_FUNCTION = new CombineAggregateMergeFunction();

    private final List<Cursor> cursors;
    private final int estimatedCountPoints;
    private long latestTsMillis = TsColumn.DEFAULT_VALUE;
    private final MergeFunction mergeFunction;

    private MergingAggrGraphDataIterator(int columnSetMask, int estimatedCountPoints, List<Cursor> cursors, MergeFunction mergeFunction) {
        super(columnSetMask);
        this.estimatedCountPoints = estimatedCountPoints;
        this.cursors = cursors;
        this.mergeFunction = mergeFunction;
    }

    public static AggrGraphDataListIterator ofMaxAggregate(List<AggrGraphDataListIterator> iterators) {
        return of(iterators, MAX_AGGREGATE_FUNCTION);
    }

    public static AggrGraphDataListIterator ofCombineAggregate(List<AggrGraphDataListIterator> iterators) {
        return of(iterators, COMBINE_AGGREGATE_FUNCTION);
    }

    public static AggrGraphDataListIterator of(List<AggrGraphDataListIterator> iterators, MergeFunction mergeFunction) {
        if (iterators.isEmpty()) {
            return EmptyAggrGraphDataIterator.INSTANCE;
        } else if (iterators.size() == 1) {
            return iterators.get(0);
        }

        int mask = 0;
        int maxPoints = -1;
        List<Cursor> cursors = new ArrayList<>(iterators.size());
        for (AggrGraphDataListIterator it : iterators) {
            mask |= it.columnSetMask();
            maxPoints = Math.max(maxPoints, it.estimatePointsCount());
            cursors.add(new Cursor(it, mergeFunction));
        }

        // reverse to use latest pushed value at same timestamp during merge
        Collections.reverse(cursors);
        return new MergingAggrGraphDataIterator(mask, maxPoints, cursors, mergeFunction);
    }

    @Override
    public boolean next(AggrPoint target) {
        boolean result = false;
        target.tsMillis = TsColumn.DEFAULT_VALUE;
        for (int index = 0; index < cursors.size(); index++) {
            result |= cursors.get(index).hasNext(target, latestTsMillis);
        }

        if (result) {
            latestTsMillis = target.tsMillis;
        }

        return result;
    }

    @Override
    public int estimatePointsCount() {
        return estimatedCountPoints;
    }

    private static class Cursor {
        private final AggrGraphDataListIterator iterator;
        private final AggrPoint point;
        private final MergeFunction mergeFunction;

        Cursor(AggrGraphDataListIterator iterator, MergeFunction mergeFunction) {
            this.iterator = iterator;
            this.mergeFunction = mergeFunction;
            this.point = new AggrPoint();
        }

        public boolean hasNext(AggrPoint target, long latestTsMillis) {
            if (point.getTsMillis() <= latestTsMillis) {
                if (!iterator.next(point)) {
                    point.tsMillis = TsColumn.DEFAULT_VALUE;
                    return false;
                }
            }

            if (target.getTsMillis() == TsColumn.DEFAULT_VALUE) {
                point.copyTo(target);
                return true;
            }

            if (target.getTsMillis() > point.getTsMillis()) {
                point.copyTo(target);
                return true;
            }

            if (target.getTsMillis() == point.getTsMillis()) {
                mergeFunction.merge(target, point);
                return true;
            }

            return true;
        }
    }

    public interface MergeFunction {
        /**
         * @param target point that will contain merged result after apply
         * @param point point that should me merged with target point
         */
        void merge(AggrPoint target, AggrPoint point);
    }

    public static class MaxAggregateMergeFunction implements MergeFunction {
        @Override
        public void merge(AggrPoint target, AggrPoint point) {
            if (point.getCount() > target.getCount()) {
                point.copyTo(target);
            }
        }
    }

    public static class CombineAggregateMergeFunction implements MergeFunction {
        @Override
        public void merge(AggrPoint target, AggrPoint point) {
            if (target.isMerge()) {
                target.tsMillis = TsColumn.merge(target.tsMillis, point.tsMillis);
                target.valueNum = ValueColumn.merge0(target.valueNum, target.valueDenom, point.valueNum, point.valueDenom);
                target.valueDenom = ValueColumn.merge1(target.valueNum, target.valueDenom, point.valueNum, point.valueDenom);
                target.merge = MergeColumn.merge(target.merge, point.merge);
                target.count = CountColumn.merge(target.count, point.count);
                target.stepMillis = StepColumn.merge(target.stepMillis, point.stepMillis);
                target.logHistogram = LogHistogramColumn.merge(target.logHistogram, point.logHistogram);
                target.histogram = HistogramColumn.merge(target.histogram, point.histogram);
                target.summaryInt64 = SummaryInt64Column.merge(target.summaryInt64, point.summaryInt64);
                target.summaryDouble = SummaryDoubleColumn.merge(target.summaryDouble, point.summaryDouble);
                target.longValue = LongValueColumn.merge(target.longValue, point.longValue);
            }
        }
    }
}
