package ru.yandex.solomon.model.timeseries;

import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;

import ru.yandex.monlib.metrics.MetricType;

import static ru.yandex.monlib.metrics.MetricType.COUNTER;
import static ru.yandex.monlib.metrics.MetricType.DGAUGE;
import static ru.yandex.monlib.metrics.MetricType.DSUMMARY;
import static ru.yandex.monlib.metrics.MetricType.HIST;
import static ru.yandex.monlib.metrics.MetricType.HIST_RATE;
import static ru.yandex.monlib.metrics.MetricType.IGAUGE;
import static ru.yandex.monlib.metrics.MetricType.ISUMMARY;
import static ru.yandex.monlib.metrics.MetricType.LOG_HISTOGRAM;
import static ru.yandex.monlib.metrics.MetricType.RATE;
import static ru.yandex.monlib.metrics.MetricType.UNKNOWN;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class MetricTypeCasts {
    private enum MetricCategory {
        UNKNOWN_CATEGORY(UNKNOWN),
        GAUGE(COUNTER, RATE, IGAUGE, DGAUGE),
        HISTOGRAM(HIST, LOG_HISTOGRAM, HIST_RATE),
        SUMMARY(ISUMMARY, DSUMMARY),
        ;

        private List<MetricType> types;

        MetricCategory(MetricType... types) {
            this.types = ImmutableList.copyOf(types);
        }

        public List<MetricType> getTypes() {
            return types;
        }
    }

    private static EnumMap<MetricType, MetricCategory> CATEGORY;
    static {
        CATEGORY = new EnumMap<MetricType, MetricCategory>(
            Arrays.stream(MetricCategory.values())
                .flatMap(mc -> mc.getTypes().stream()
                    .map(mt -> Map.entry(mt, mc))
                )
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue))
        );
    }

    /**
     * @return Common metric type or UNKNOWN if no common type exists
     */
    public static MetricType commonType(MetricType left, MetricType right) {
        if (left == right) {
            return left;
        }

        var leftCat = CATEGORY.get(left);
        var rightCat = CATEGORY.get(right);

        if (leftCat != rightCat) {
            return UNKNOWN;
        }

        switch (leftCat) {
            case UNKNOWN_CATEGORY:
                return UNKNOWN;
            case GAUGE:
                return commonGauge(left, right);
            case HISTOGRAM:
                return commonHist(left, right);
            case SUMMARY:
                return commonSummary(left, right);
            default:
                throw new IllegalStateException("Unknown metric category: " + leftCat);
        }
    }

    private static MetricType commonGauge(MetricType left, MetricType right) {
        if (left == right) {
            return left;
        }
        if (left == DGAUGE || right == DGAUGE) {
            return DGAUGE;
        }
        if (left == COUNTER) {
            return right;
        }
        if (right == COUNTER) {
            return left;
        }
        if (left == RATE && right == IGAUGE) {
            return DGAUGE;
        }
        if (left == IGAUGE && right == RATE) {
            return DGAUGE;
        }
        throw new IllegalStateException("Uncovered case: " + left + " and " + right);
    }

    private static MetricType commonHist(MetricType left, MetricType right) {
        if (left == right) {
            return left;
        }
        if (left == HIST_RATE || right == HIST_RATE) {
            return HIST_RATE;
        }
        if (left == HIST && right == LOG_HISTOGRAM) {
            return HIST;
        }
        if (left == LOG_HISTOGRAM && right == HIST) {
            return HIST;
        }
        throw new IllegalStateException("Uncovered case: " + left + " and " + right);
    }

    private static MetricType commonSummary(MetricType left, MetricType right) {
        if (left == right) {
            return left;
        }
        if (left == DSUMMARY || right == DSUMMARY) {
            return DSUMMARY;
        }
        throw new IllegalStateException("Uncovered case: " + left + " and " + right);
    }
}
