package ru.yandex.solomon.model.timeseries;

import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.protobuf.MetricType;

import static ru.yandex.solomon.model.point.column.StockpileColumns.minColumnSet;
import static ru.yandex.solomon.model.timeseries.MetricTypes.isPrimitive;

/**
 * <pre>
 *    F/T   |  DGAUGE   | IGAUGE  | COUNTER |   RATE      |   LOG_HIST  |     HIST    |
 *  -----------------------------------------------------------------------------------
 *  DGAUGE  |   same    |  round  |  round  |  integrate  |     none    |     none    |
 *  -----------------------------------------------------------------------------------
 *  IGAUGE  |   cast    |  same   |  same   |  same       |     none    |     none    |
 *  -----------------------------------------------------------------------------------
 *  COUNTER |   cast    |  same   |  same   |  same       |     none    |     none    |
 *  -----------------------------------------------------------------------------------
 *  RATE    |   deriv   |  same   |  same   |  same       |     none    |     none    |
 *  -----------------------------------------------------------------------------------
 *  LOG_HIST|   none    |  none   |  none   |  none       |     same    |    convert  |
 *  -----------------------------------------------------------------------------------
 *  HIST    |   none    |  none   |  none   |  none       |    convert  |     same    |
 * </pre>
 *
 * @author Vladimir Gordiychuk
 */
public class MetricTypeTransfers {
    public static AggrGraphDataListIterator of(MetricType from, MetricType to, AggrGraphDataListIterator source) {
        if (from == to) {
            return source;
        }

        int mask = refreshMask(from, to, source.columnSetMask());
        switch (from) {
            case DGAUGE:
                return ofGaugeDouble(mask, to, source);
            case IGAUGE:
                return ofGaugeInt64(mask, to, source);
            case COUNTER:
                return ofCounter(mask, to, source);
            case RATE:
                return ofRate(mask, to, source);
            case LOG_HISTOGRAM:
                return ofLogHistogram(mask, to, source);
            case HIST:
                return ofHistogram(mask, to, source);
            default:
                throw new UnsupportedOperationException("Not able convert from " + from + " to " + to);
        }
    }

    public static boolean isAvailableTransfer(MetricType from, MetricType to) {
        if (from == to) {
            return true;
        }

        if (from == MetricType.LOG_HISTOGRAM) {
            return to == MetricType.HIST;
        } else if (from == MetricType.HIST) {
            return to == MetricType.LOG_HISTOGRAM;
        }

        return isPrimitive(from) && isPrimitive(to);
    }

    private static AggrGraphDataListIterator ofGaugeDouble(int mask, MetricType to, AggrGraphDataListIterator source) {
        switch (to) {
            case DGAUGE:
                return source;
            case IGAUGE:
            case COUNTER:
                return TransferIterator.of(mask, source, point -> {
                    point.longValue = Math.round(point.getValueDivided());
                });
            case RATE:
                return IntegrateAggrGraphDataIterator.of(source);
            default:
                throw throwUnsupported(MetricType.DGAUGE, to);
        }
    }

    private static AggrGraphDataListIterator ofGaugeInt64(int mask, MetricType to, AggrGraphDataListIterator source) {
        switch (to) {
            case DGAUGE:
                return TransferIterator.of(mask, source, point -> {
                    point.valueNum = point.longValue;
                    point.valueDenom = ValueColumn.DEFAULT_DENOM;
                });
            case IGAUGE:
            case COUNTER:
            case RATE:
                return source;
            default:
                throw throwUnsupported(MetricType.IGAUGE, to);
        }
    }

    private static AggrGraphDataListIterator ofCounter(int mask, MetricType to, AggrGraphDataListIterator source) {
        switch (to) {
            case RATE:
            case IGAUGE:
            case COUNTER:
                return source;
            case DGAUGE:
                return TransferIterator.of(mask, source, point -> {
                    point.valueNum = point.longValue;
                    point.valueDenom = ValueColumn.DEFAULT_DENOM;
                });
            default:
                throw throwUnsupported(MetricType.COUNTER, to);
        }
    }

    private static AggrGraphDataListIterator ofRate(int mask, MetricType to, AggrGraphDataListIterator source) {
        switch (to) {
            case DGAUGE:
                return DerivAggrGraphDataIterator.of(source);
            case RATE:
            case IGAUGE:
            case COUNTER:
                return source;
            default:
                throw throwUnsupported(MetricType.RATE, to);
        }
    }

    private static AggrGraphDataListIterator ofLogHistogram(int mask, MetricType to, AggrGraphDataListIterator source) {
        switch (to) {
            case LOG_HISTOGRAM:
                return source;
            case HIST:
                return TransferLogHistogramToUgramIterator.of(mask, source);
            default:
                throw throwUnsupported(MetricType.LOG_HISTOGRAM, to);
        }
    }

    private static AggrGraphDataListIterator ofHistogram(int mask, MetricType to, AggrGraphDataListIterator source) {
        switch (to) {
            case HIST:
                return source;
            case LOG_HISTOGRAM:
                return new TransferHistToLogHistogramIterator(mask, source);
            default:
                throw throwUnsupported(MetricType.HIST, to);
        }
    }

    private static UnsupportedOperationException throwUnsupported(MetricType from, MetricType to) {
        throw new UnsupportedOperationException("Not able convert from " + from + " to " + to);
    }

    private static int refreshMask(MetricType from, MetricType to, int sourceMask) {
        return (sourceMask & ~minColumnSet(from)) | minColumnSet(to);
    }

    @ParametersAreNonnullByDefault
    private static class TransferIterator extends AggrGraphDataListIterator {
        private final AggrGraphDataListIterator source;
        private final Consumer<AggrPoint> fn;

        public TransferIterator(int columnSetMask, AggrGraphDataListIterator source, Consumer<AggrPoint> fn) {
            super(columnSetMask);
            this.source = source;
            this.fn = fn;
        }

        private static AggrGraphDataListIterator of(int mask, AggrGraphDataListIterator source, Consumer<AggrPoint> fn) {
            return new TransferIterator(mask, source, fn);
        }

        @Override
        public int estimatePointsCount() {
            return source.estimatePointsCount();
        }

        @Override
        public boolean next(AggrPoint target) {
            if (!source.next(target)) {
                return false;
            }

            fn.accept(target);
            target.columnSet = columnSet;
            return true;
        }
    }
}
