package ru.yandex.solomon.model.timeseries;

import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.model.point.AggrPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class TimeFilterAggrGraphDataIterator {

    public static AggrGraphDataListIterator sliceFrom(AggrGraphDataListIterator source, long fromTsMillis) {
        return new From(source, fromTsMillis);
    }

    public static AggrGraphDataListIterator sliceTo(AggrGraphDataListIterator source, long toTsMillis) {
        return new To(source, toTsMillis);
    }

    public static AggrGraphDataListIterator slice(AggrGraphDataListIterator source, long fromTsMillis, long toTsMillis) {
        return sliceTo(sliceFrom(source, fromTsMillis), toTsMillis);
    }

    public static AggrGraphDataIterable slice(AggrGraphDataIterable source, long fromTsMillis, long toTsMillis) {
        return new Iterable(source, iterator -> slice(iterator, fromTsMillis, toTsMillis));
    }

    @ParametersAreNonnullByDefault
    private static class From extends AggrGraphDataListIterator {
        private AggrGraphDataListIterator source;
        private long fromIncluded;

        public From(AggrGraphDataListIterator source, long fromIncluded) {
            super(source.columnSetMask());
            this.source = source;
            this.fromIncluded = fromIncluded;
        }

        @Override
        public int estimatePointsCount() {
            return source.estimatePointsCount();
        }

        @Override
        public boolean next(AggrPoint target) {
            while (source.next(target)) {
                if (target.tsMillis >= fromIncluded) {
                    return true;
                }
            }

            return false;
        }
    }

    @ParametersAreNonnullByDefault
    private static class To extends AggrGraphDataListIterator {
        private AggrGraphDataListIterator source;
        private long endIncluded;

        public To(AggrGraphDataListIterator source, long endIncluded) {
            super(source.columnSetMask());
            this.source = source;
            this.endIncluded = endIncluded;
        }

        @Override
        public int estimatePointsCount() {
            return source.estimatePointsCount();
        }

        @Override
        public boolean next(AggrPoint target) {
            if (source.next(target)) {
                return target.tsMillis <= endIncluded;
            }

            return false;
        }
    }

    public static class Iterable implements AggrGraphDataIterable {
        private AggrGraphDataIterable source;
        private final Function<AggrGraphDataListIterator, AggrGraphDataListIterator> fn;

        public Iterable(AggrGraphDataIterable source, Function<AggrGraphDataListIterator, AggrGraphDataListIterator> fn) {
            this.source = source;
            this.fn = fn;
        }

        @Override
        public int getRecordCount() {
            return source.getRecordCount();
        }

        @Override
        public int elapsedBytes() {
            return source.elapsedBytes();
        }

        @Override
        public AggrGraphDataListIterator iterator() {
            return fn.apply(source.iterator());
        }

        @Override
        public int columnSetMask() {
            return source.columnSetMask();
        }
    }
}
