package ru.yandex.solomon.model.timeseries;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import it.unimi.dsi.fastutil.longs.LongArrayList;

import ru.yandex.solomon.util.collection.array.LongArrayView;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class Timeline {

    // Sorted, no dups.
    @Nonnull
    private final LongArrayView timestamps;

    public Timeline() {
        this.timestamps = LongArrayView.empty;
    }

    public Timeline(long[] timestamps, SortedOrCheck sorted) {
        this(new LongArrayView(timestamps), sorted);
    }

    public Timeline(LongArrayView timestamps, SortedOrCheck sorted) {
        this.timestamps = timestamps;
        if (sorted == SortedOrCheck.CHECK) {
            if (!isSortedUnique(timestamps)) {
                throw new IllegalArgumentException("timestamps are not sorted unique yet");
            }
        }
    }

    public boolean isSortedUnique(LongArrayView timestamps) {
        for (int i = 1; i < timestamps.length(); ++i) {
            if (timestamps.at(i) <= timestamps.at(i - 1)) {
                return false;
            }
        }
        return true;
    }

    public Timeline(Interval interval, int points) {
        if (points == 0) {
            timestamps = LongArrayView.empty;
        } else if (points == 1) {
            timestamps = new LongArrayView(new long[]{interval.getBeginMillis()});
        } else {
            long step = (interval.getEndMillis() - interval.getBeginMillis()) / (points - 1);
            long[] timestamps = new long[points];
            timestamps[0] = interval.getBeginMillis();
            for (int i = 1; i < points; i++) {
                timestamps[i] = timestamps[i - 1] + step;
            }
            this.timestamps = new LongArrayView(timestamps);
        }
    }

    public int getPointCount() {
        return timestamps.length();
    }

    public LongArrayView getPointsMillis() {
        return timestamps;
    }

    public long getPointMillisAt(int pos) {
        return timestamps.at(pos);
    }

    public long lengthAt(int pos) {
        if (timestamps.length() == 1) {
            return 0;
        }

        if (pos == 0) {
            return timestamps.at(1) - timestamps.at(0);
        }
        if (pos == timestamps.length() - 1) {
            return timestamps.at(timestamps.length() - 1) - timestamps.at(timestamps.length() - 2);
        }

        return timestamps.at(pos + 1) - timestamps.at(pos - 1);
    }

    public Timeline dropStepsShorterThan(long maxStepMillis) {
        if (timestamps.length() <= 1) {
            return this;
        }

        LongArrayList r = new LongArrayList();

        long lastPoint = timestamps.at(0);
        r.add(lastPoint);

        for (int i = 1; i < timestamps.length(); ++i) {
            long point = timestamps.at(i);
            if (point - lastPoint >= maxStepMillis || i == timestamps.length() - 1) {
                lastPoint = point;
                r.add(lastPoint);
            }
        }

        return new Timeline(r.toLongArray(), SortedOrCheck.SORTED_UNIQUE);
    }

    public long first() {
        return timestamps.first();
    }

    public long last() {
        return timestamps.at(timestamps.length() - 1);
    }

    public long averageStepMillis() {
        if (timestamps.length() <= 1) {
            return 0;
        }
        return (last() - first()) / (timestamps.length() - 1);
    }

    public Interval interval() {
        return Interval.millis(first(), last());
    }

    public boolean isIn(Interval interval) {
        if (timestamps.length() == 0) {
            return true;
        }

        return this.first() >= interval.getBeginMillis()
                && this.last() <= interval.getEndMillis();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Timeline timeline = (Timeline) o;

        return timestamps.equals(timeline.timestamps);
    }

    @Override
    public int hashCode() {
        return timestamps.hashCode();
    }

    @Override
    public String toString() {
        return "Timeline(" + timestamps + ")";
    }

    public int length() {
        return timestamps.length();
    }

    public boolean isEmpty() {
        return timestamps.isEmpty();
    }

    public class Navigator {
        private int pos;
        private long scrolledToMillis = 0;

        public long currentMillis() {
            return timestamps.at(pos);
        }

        public void scrollBackwardMillis(long millis) {
            if (millis > scrolledToMillis) {
                throw new IllegalArgumentException("scroll forward is not allowed");
            }
            if (timestamps.length() == 0) {
                this.scrolledToMillis = millis;
                return;
            }
            for (; ; ) {
                if (pos == 0) {
                    break;
                }
                if (timestamps.at(pos - 1) < millis) {
                    break;
                }
                --pos;
            }

            this.scrolledToMillis = millis;
        }

        public void scrollToMillis(long millis) {
            if (millis < scrolledToMillis) {
                scrollBackwardMillis(millis);
                return;
            }

            if (timestamps.length() == 0) {
                this.scrolledToMillis = millis;
                return;
            }

            for (;;) {
                if (pos == timestamps.length() - 1) {
                    break;
                }
                if (timestamps.at(pos + 1) > millis) {
                    break;
                }
                ++pos;
            }

            this.scrolledToMillis = millis;
        }

        public int pos() {
            return pos;
        }

        public boolean isOnPoint() {
            return timestamps.length() != 0 && timestamps.at(pos) == scrolledToMillis;
        }

        public boolean isBetweenPoints() {
            if (timestamps.length() == 0) {
                return false;
            }
            if (isOnPoint()) {
                return false;
            }
            if (pos == 0 && scrolledToMillis < timestamps.at(0)) {
                return false;
            }
            return pos < timestamps.length() - 1;
        }
    }


    public static Timeline union(Timeline a, Timeline b) {
        if (a.timestamps.length() == 0) {
            return b;
        }
        if (b.timestamps.length() == 0) {
            return a;
        }

        if (a.equals(b)) {
            return a;
        }

        return union(List.of(a, b));
    }

    public static Timeline union(List<Timeline> timelines) {
        if (timelines.size() == 0) {
            return new Timeline();
        }

        if (timelines.size() == 1) {
            return timelines.get(0);
        }

        int capacity = 0;
        List<Cursor> cursors = new ArrayList<>(timelines.size());
        for (var timeline : timelines) {
            if (timeline.length() == 0) {
                continue;
            }

            cursors.add(new Cursor(timeline));
            capacity = Math.max(capacity, timeline.length());
        }

        long prev = 0;
        long min;
        LongArrayList timestamps = new LongArrayList();
        while (true) {
            min = Long.MAX_VALUE;
            for (var cursor : cursors) {
                if (cursor.current <= prev && !cursor.next()) {
                    continue;
                }
                min = Long.min(min, cursor.current);
            }

            if (min != Long.MAX_VALUE) {
                prev = min;
                timestamps.add(min);
            } else {
                break;
            }
        }
        return new Timeline(new LongArrayView(timestamps.elements(), 0, timestamps.size()), SortedOrCheck.SORTED_UNIQUE);
    }

    private static class Cursor {
        private final Timeline timeline;
        private int pos;
        private long current = 0;

        Cursor(Timeline timeline) {
            this.timeline = timeline;
            this.pos = 0;
        }

        public boolean next() {
            if (pos >= timeline.length()) {
                return false;
            }

            current = timeline.getPointMillisAt(pos++);
            return true;
        }
    }
}
