package ru.yandex.solomon.model.timeseries;

import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.model.type.MutableLogHistogram;
import ru.yandex.solomon.model.type.ugram.Ugram;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class TransferLogHistogramToUgramIterator extends AggrGraphDataListIterator {
    private final static long STEP_MILLIS = TimeUnit.MINUTES.toMillis(5L);

    private final AggrGraphDataListIterator source;
    private final Ugram ugram;
    private final AggrGraphDataArrayList buffer;
    private final RecyclableAggrPoint tmp;

    private final MutableLogHistogram mutableLog;
    private final LogHistogram mergedLog;
    private final Histogram mergedHist;

    private boolean oneMore;
    private int bufferIdx;
    private boolean done;

    private TransferLogHistogramToUgramIterator(int mask, AggrGraphDataListIterator source) {
        super(mask);
        this.source = source;
        this.ugram = Ugram.create();
        this.buffer = new AggrGraphDataArrayList(source.columnSetMask(), 100);
        this.tmp = RecyclableAggrPoint.newInstance();
        this.mergedLog = LogHistogram.newInstance();
        this.mergedHist = Histogram.newInstance();
        this.mutableLog = new MutableLogHistogram();
    }

    public static AggrGraphDataListIterator of(int mask, AggrGraphDataListIterator source) {
        return new TransferLogHistogramToUgramIterator(mask, source);
    }

    @Override
    public int estimatePointsCount() {
        return source.estimatePointsCount();
    }

    @Override
    public boolean next(AggrPoint target) {
        if (done) {
            return false;
        }

        if (nextFromBuffer(target)) {
            return true;
        } else {
            fillBuffer();
            if (nextFromBuffer(target)) {
                return true;
            } else {
                done = true;
                tmp.recycle();
                mergedLog.recycle();
                mergedHist.recycle();
                ugram.recycle();
                return false;
            }
        }
    }

    private boolean nextFromBuffer(AggrPoint target) {
        if (buffer.isEmpty()) {
            return false;
        }

        if (buffer.length() == 1) {
            buffer.getDataTo(0, target);
            buffer.clear();
            target.histogram = Histogram.orNew(target.histogram);
            target.columnSet = columnSetMask();
            convert(target.logHistogram, target.histogram);
            return true;
        }

        if (bufferIdx == 0) {
            alightBuffer();
        }

        buffer.getDataTo(bufferIdx++, target);
        target.histogram = Histogram.orNew(target.histogram);
        target.columnSet = columnSetMask();
        convertAlighted(target.logHistogram, target.histogram);
        if (bufferIdx >= buffer.length()) {
            bufferIdx = 0;
            buffer.clear();
        }
        return true;
    }

    private void fillBuffer() {
        if (!oneMore && !source.next(tmp)) {
            return;
        }

        long windowStart = tmp.tsMillis - (tmp.tsMillis % STEP_MILLIS);
        long windowEnd = windowStart + STEP_MILLIS;
        buffer.addRecordData(source.columnSet, tmp);
        oneMore = false;

        while (source.next(tmp)) {
            if (tmp.tsMillis < windowEnd) {
                buffer.addRecordData(source.columnSet, tmp);
            } else {
                oneMore = true;
                break;
            }
        }
    }

    private void convert(@Nullable LogHistogram from, Histogram to) {
        if (from == null) {
            return;
        }

        int targetSize = 0;
        if (from.getCountZero() != 0) {
            to.setUpperBound(targetSize, 0.01);
            to.setBucketValue(targetSize, from.getCountZero());
            targetSize++;
        }

        boolean useUgram = false;
        for (int index = 0; index < from.countBucket(); index++) {
            double lowerBound = upperBound(from.getBase(), from.getStartPower() + index);
            double upperBound = upperBound(from.getBase(), from.getStartPower() + index + 1);

            if (targetSize + 2 >= Histograms.MAX_BUCKETS_COUNT) {
                ugram.merge(to);
                to.reset();
                useUgram = true;
                targetSize = 0;
            }

            if (targetSize == 0 || to.upperBound(targetSize - 1) != lowerBound) {
                to.setUpperBound(targetSize, lowerBound);
                to.setBucketValue(targetSize, 0);
                targetSize++;
            }

            to.setUpperBound(targetSize, upperBound);
            to.setBucketValue(targetSize, Math.round(from.getBucketValue(index)));
            targetSize++;
        }

        if (useUgram) {
            ugram.merge(to);
            ugram.snapshot(to);
            ugram.reset();
        }
    }

    private void convertAlighted(@Nullable LogHistogram from, Histogram to) {
        if (from == null) {
            return;
        }

        for (int index =0; index < mergedHist.count(); index++) {
            to.setUpperBound(index, mergedHist.upperBound(index));
        }

        if (from.getCountZero() != 0) {
            to.addValue(0, from.getCountZero());
        }

        for (int index = 0; index < from.countBucket(); index++) {
            long count = Math.round(from.getBucketValue(index));
            if (count == 0) {
                continue;
            }
            double value = upperBound(from.getBase(), from.getStartPower() + index + 1) - 0.01;
            to.addValue(value, count);
        }
    }

    private void alightBuffer() {
        mutableLog.reset();
        mergedHist.reset();
        mergedLog.reset();
        for (int index = 0; index < buffer.length(); index++) {
            mutableLog.addHistogram(buffer.getLogHistogram(index));
        }
        mutableLog.build(mergedLog);
        convert(mergedLog, mergedHist);
    }

    private double upperBound(double base, int idx) {
        return Math.pow(base, idx);
    }
}
