package ru.yandex.solomon.model.timeseries.aggregation;

import javax.annotation.Nullable;

import ru.yandex.solomon.math.protobuf.Aggregate;
import ru.yandex.solomon.math.protobuf.DoubleAggregate;
import ru.yandex.solomon.math.protobuf.HistogramAggregate;
import ru.yandex.solomon.math.protobuf.Int64Aggregate;
import ru.yandex.solomon.math.protobuf.LogHistogramAggregate;
import ru.yandex.solomon.math.protobuf.SummaryDoubleAggregate;
import ru.yandex.solomon.math.protobuf.SummaryInt64Aggregate;
import ru.yandex.solomon.model.protobuf.HistogramConverter;
import ru.yandex.solomon.model.protobuf.LogHistogramConverter;
import ru.yandex.solomon.model.protobuf.SummaryConverter;
import ru.yandex.stockpile.api.MetricData;

/**
 * @author Vladimir Gordiychuk
 */
public final class AggregateConverters {

    public static MetricData.Builder fillAggregate(MetricData.Builder builder, TimeseriesSummary aggregate) {
        if (aggregate instanceof DoubleSummary) {
            return builder.setDouble(toProtoAggregate((DoubleSummary) aggregate));
        }

        if (aggregate instanceof Int64Summary) {
            return builder.setInt64(toProtoAggregate((Int64Summary) aggregate));
        }

        if (aggregate instanceof HistogramSummary) {
            return builder.setHistogram(toProtoAggregate((HistogramSummary) aggregate));
        }

        if (aggregate instanceof SummaryDoubleSummary) {
            return builder.setSummaryDouble(toProtoAggregate((SummaryDoubleSummary) aggregate));
        }

        if (aggregate instanceof SummaryInt64Summary) {
            return builder.setSummaryInt64(toProtoAggregate((SummaryInt64Summary) aggregate));
        }

        throw new UnsupportedOperationException("Unsupported aggregate type: " + aggregate);
    }

    public static Aggregate toProto(TimeseriesSummary aggregate) {
        if (aggregate instanceof DoubleSummary) {
            return toProto((DoubleSummary) aggregate);
        }

        if (aggregate instanceof Int64Summary) {
            return toProto((Int64Summary) aggregate);
        }

        if (aggregate instanceof HistogramSummary) {
            return toProto((HistogramSummary) aggregate);
        }

        if (aggregate instanceof SummaryDoubleSummary) {
            return toProto((SummaryDoubleSummary) aggregate);
        }

        if (aggregate instanceof SummaryInt64Summary) {
            return toProto((SummaryInt64Summary) aggregate);
        }

        throw new UnsupportedOperationException("Unsupported aggregate type: " + aggregate);
    }

    public static Aggregate toProto(DoubleSummary aggerega) {
        return Aggregate.newBuilder()
                .setDouble(toProtoAggregate(aggerega))
                .build();
    }

    public static Aggregate toProto(Int64Summary aggregate) {
        return Aggregate.newBuilder()
            .setInt64(toProtoAggregate(aggregate))
            .build();
    }

    public static DoubleAggregate toProtoAggregate(DoubleSummary aggregate) {
        return DoubleAggregate.newBuilder()
                .setMax(aggregate.getMax())
                .setMin(aggregate.getMin())
                .setSum(aggregate.getSum())
                .setAvg(aggregate.getAvg())
                .setLast(aggregate.getLast())
                .setCount(aggregate.getCount())
                .setMask(aggregate.getMask())
                .build();
    }

    public static Int64Aggregate toProtoAggregate(Int64Summary aggregate) {
        return Int64Aggregate.newBuilder()
            .setMax(aggregate.getMax())
            .setMin(aggregate.getMin())
            .setSum(aggregate.getSum())
            .setAvg(aggregate.getAvg())
            .setLast(aggregate.getLast())
            .setCount(aggregate.getCount())
            .setMask(aggregate.getMask())
            .build();
    }

    public static Aggregate toProto(HistogramSummary aggregate) {
        return Aggregate.newBuilder()
                .setHistogram(toProtoAggregate(aggregate))
                .build();
    }

    public static HistogramAggregate toProtoAggregate(HistogramSummary aggregate) {
        return HistogramAggregate.newBuilder()
                .setSum(HistogramConverter.toProto(aggregate.getSum()))
                .setLast(HistogramConverter.toProto(aggregate.getLast()))
                .setCount(aggregate.getCount())
                .setMask(aggregate.getMask())
                .build();
    }

    public static Aggregate toProto(SummaryDoubleSummary aggregate) {
        return Aggregate.newBuilder()
                .setSummaryDouble(toProtoAggregate(aggregate))
                .build();
    }

    public static SummaryDoubleAggregate toProtoAggregate(SummaryDoubleSummary aggregate) {
        return SummaryDoubleAggregate.newBuilder()
                .setSum(SummaryConverter.toProto(aggregate.getSum()))
                .setLast(SummaryConverter.toProto(aggregate.getLast()))
                .setCount(aggregate.getCount())
                .setMask(aggregate.getMask())
                .build();
    }

    public static Aggregate toProto(SummaryInt64Summary aggregate) {
        return Aggregate.newBuilder()
                .setSummaryInt64(toProtoAggregate(aggregate))
                .build();
    }

    public static SummaryInt64Aggregate toProtoAggregate(SummaryInt64Summary aggregate) {
        return SummaryInt64Aggregate.newBuilder()
                .setSum(SummaryConverter.toProto(aggregate.getSum()))
                .setLast(SummaryConverter.toProto(aggregate.getLast()))
                .setCount(aggregate.getCount())
                .setMask(aggregate.getMask())
                .build();
    }

    public static TimeseriesSummary fromProto(Aggregate aggregate) {
        switch (aggregate.getTypeCase()) {
            case DOUBLE:
                return fromProto(aggregate.getDouble());
            case INT64:
                return fromProto(aggregate.getInt64());
            case HISTOGRAM:
                return fromProto(aggregate.getHistogram());
            case LOG_HISTOGRAM:
                return fromProto(aggregate.getLogHistogram());
            case SUMMARY_DOUBLE:
                return fromProto(aggregate.getSummaryDouble());
            case SUMMARY_INT64:
                return fromProto(aggregate.getSummaryInt64());
            default:
                throw new UnsupportedOperationException("Unsupported aggregate: " + aggregate);
        }
    }

    @Nullable
    public static TimeseriesSummary fromProto(MetricData proto) {
        switch (proto.getAggregateCase()) {
            case AGGREGATE_NOT_SET:
                return null;
            case DOUBLE:
                return fromProto(proto.getDouble());
            case INT64:
                return fromProto(proto.getInt64());
            case HISTOGRAM:
                return fromProto(proto.getHistogram());
            case LOG_HISTOGRAM:
                return fromProto(proto.getLogHistogram());
            case SUMMARY_DOUBLE:
                return fromProto(proto.getSummaryDouble());
            case SUMMARY_INT64:
                return fromProto(proto.getSummaryInt64());
            default:
                throw new UnsupportedOperationException("Unsupported aggregate: " + proto.getAggregateCase());
        }
    }

    public static DoubleSummary fromProto(DoubleAggregate proto) {
        DoubleSummary result = new DoubleSummary();
        result.setMax(proto.getMax());
        result.setMin(proto.getMin());
        result.setSum(proto.getSum());
        result.setAvg(proto.getAvg());
        result.setLast(proto.getLast());
        result.setCount(proto.getCount());
        result.setMask(proto.getMask());
        return result;
    }

    public static Int64Summary fromProto(Int64Aggregate proto) {
        Int64Summary result = new Int64Summary();
        result.setMax(proto.getMax());
        result.setMin(proto.getMin());
        result.setSum(proto.getSum());
        result.setAvg(proto.getAvg());
        result.setLast(proto.getLast());
        result.setCount(proto.getCount());
        result.setMask(proto.getMask());
        return result;
    }

    public static HistogramSummary fromProto(HistogramAggregate proto) {
        HistogramSummary result = new HistogramSummary();
        result.setSum(HistogramConverter.fromProto(proto.getSum()));
        result.setLast(HistogramConverter.fromProto(proto.getLast()));
        result.setCount(proto.getCount());
        result.setMask(proto.getMask());
        return result;
    }

    public static LogHistogramSummary fromProto(LogHistogramAggregate proto) {
        LogHistogramSummary result = new LogHistogramSummary();
        result.setSum(LogHistogramConverter.fromProto(proto.getSum()));
        result.setLast(LogHistogramConverter.fromProto(proto.getLast()));
        result.setCount(proto.getCount());
        result.setMask(proto.getMask());
        return result;
    }

    public static SummaryDoubleSummary fromProto(SummaryDoubleAggregate proto) {
        SummaryDoubleSummary result = new SummaryDoubleSummary();
        result.setSum(SummaryConverter.fromProto(proto.getSum()));
        result.setLast(SummaryConverter.fromProto(proto.getLast()));
        result.setCount(proto.getCount());
        result.setMask(proto.getMask());
        return result;
    }

    public static SummaryInt64Summary fromProto(SummaryInt64Aggregate proto) {
        SummaryInt64Summary result = new SummaryInt64Summary();
        result.setSum(SummaryConverter.fromProto(proto.getSum()));
        result.setLast(SummaryConverter.fromProto(proto.getLast()));
        result.setCount(proto.getCount());
        result.setMask(proto.getMask());
        return result;
    }
}
