package ru.yandex.solomon.model.timeseries.aggregation.collectors;

import java.util.function.DoubleBinaryOperator;
import java.util.function.ToDoubleFunction;

import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.ValueColumn;

/**
 * @author Vladimir Gordiychuk
 */
public class DoublePointCollectors {
    public static PointValueCollector ofDouble(Aggregation aggregation) {
        switch (aggregation) {
            case DEFAULT_AGGREGATION:
            case AVG:
                return new Collector(new DoubleStateSummary(DoubleStateSummary::getAverage));
            case LAST:
                return new Collector(new DoubleStateValue(Double.NaN, (left, right) -> right));
            case MAX:
                return new Collector(new DoubleStateValue(Double.NEGATIVE_INFINITY, Math::max));
            case MIN:
                return new Collector(new DoubleStateValue(Double.POSITIVE_INFINITY, Math::min));
            case COUNT:
                return new Collector(new DoubleStateSummary(DoubleStateSummary::getCount));
            case SUM:
                return new Collector(new DoubleStateSummary(DoubleStateSummary::getSum));
            default:
                throw new UnsupportedOperationException("Unsupported double aggregation: " + aggregation);
        }
    }

    private interface DoubleState {
        void append(double value);
        void reset();
        double compute();
    }

    private static class Collector implements PointValueCollector {
        private long count = 0;
        private DoubleState state;

        public Collector(DoubleState state) {
            this.state = state;
            this.reset();
        }

        @Override
        public void reset() {
            this.count = 0;
            this.state.reset();
        }

        @Override
        public void append(AggrPoint point) {
            double value = point.getValueDivided();
            if (Double.isNaN(value)) {
                return;
            }

            count += Math.max(1, point.count);
            state.append(value);
        }

        @Override
        public boolean compute(AggrPoint point) {
            point.setCount(count);
            if (count == 0) {
                point.setValue(Double.NaN, ValueColumn.DEFAULT_DENOM);
                return false;
            }

            point.setValue(state.compute(), ValueColumn.DEFAULT_DENOM);
            return true;
        }
    }

    private static class DoubleStateValue implements DoubleState {
        private final DoubleBinaryOperator op;
        private final double init;
        private double state;

        public DoubleStateValue(double init, DoubleBinaryOperator op) {
            this.init = init;
            this.op = op;
            this.state = init;
        }

        @Override
        public void append(double value) {
            state = op.applyAsDouble(state, value);
        }

        @Override
        public void reset() {
            state = init;
        }

        @Override
        public double compute() {
            return state;
        }
    }

    private static class DoubleStateSummary implements DoubleState {
        private final ToDoubleFunction<DoubleStateSummary> finisher;
        private long count;
        private double sum;
        private double sumCompensation; // Low order bits of sum
        private double simpleSum; // Used to compute right sum for non-finite inputs

        public DoubleStateSummary(ToDoubleFunction<DoubleStateSummary> finisher) {
            this.finisher = finisher;
        }

        @Override
        public void append(double value) {
            ++count;
            simpleSum += value;
            sumWithCompensation(value);
        }

        @Override
        public void reset() {
            count = 0;
            sum = 0;
            sumCompensation = 0;
            simpleSum = 0;
        }

        private void sumWithCompensation(double value) {
            double tmp = value - sumCompensation;
            double velvel = sum + tmp; // Little wolf of rounding error
            sumCompensation = (velvel - sum) - tmp;
            sum = velvel;
        }

        @Override
        public double compute() {
            try {
                return finisher.applyAsDouble(this);
            } finally {
                reset();
            }
        }

        final double getSum() {
            // Better error bounds to add both terms as the final sum
            double tmp = sum + sumCompensation;
            if (Double.isNaN(tmp) && Double.isInfinite(simpleSum)) {
                // If the compensated sum is spuriously NaN from
                // accumulating one or more same-signed infinite values,
                // return the correctly-signed infinity stored in
                // simpleSum.
                return simpleSum;
            } else {
                return tmp;
            }
        }

        final long getCount() {
            return count;
        }

        final double getAverage() {
            return getCount() > 0 ? getSum() / getCount() : 0.0d;
        }
    }
}
