package ru.yandex.solomon.model.timeseries.aggregation.collectors;

import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.HistogramColumn;
import ru.yandex.solomon.model.type.Histogram;
import ru.yandex.solomon.model.type.MutableHistogram;

/**
 * @author Vladimir Gordiychuk
 */
public class HistogramPointCollectors {
    public static PointValueCollector ofHistogram(Aggregation aggregation) {
        switch (aggregation) {
            case DEFAULT_AGGREGATION:
            case SUM:
                return new SumHistogramCollector();
            case LAST:
                return new LastHistogramCollector();
            default:
                throw new UnsupportedOperationException("Unsupported histogram aggregation: " + aggregation);
        }
    }

    private static class SumHistogramCollector implements PointValueCollector {
        private long count = 0;
        private MutableHistogram state = new MutableHistogram();

        @Override
        public void reset() {
            count = 0;
            state.reset();
        }

        @Override
        public void append(AggrPoint point) {
            count += Math.max(1, point.count);
            state.addHistogram(point.histogram);
        }

        @Override
        public boolean compute(AggrPoint point) {
            point.setCount(count);
            if (count == 0) {
                if (point.histogram != null) {
                    point.histogram.reset();
                }
                return false;
            }

            point.setHistogram(state.snapshot(point.histogram));
            return true;
        }
    }

    private static class LastHistogramCollector implements PointValueCollector {
        private Histogram state = Histogram.newInstance();
        private long count = 0;

        @Override
        public void reset() {
            state.reset();
            count = 0;
        }

        @Override
        public void append(AggrPoint point) {
            if (!HistogramColumn.isNullOrEmpty(point.histogram)) {
                state.copyFrom(point.histogram);
                count += Math.max(1, point.count);
            }
        }

        @Override
        public boolean compute(AggrPoint point) {
            point.setCount(count);
            if (count == 0) {
                if (point.histogram != null) {
                    point.histogram.reset();
                }
                return false;
            }

            point.setHistogram(Histogram.orNew(point.histogram).copyFrom(state));
            return true;
        }
    }
}
