package ru.yandex.solomon.model.timeseries.aggregation.collectors;

import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.LogHistogramColumn;
import ru.yandex.solomon.model.type.LogHistogram;
import ru.yandex.solomon.model.type.MutableLogHistogram;

/**
 * @author Vladimir Gordiychuk
 */
public class LogHistogramPointCollectors {
    public static PointValueCollector ofLogHistogram(Aggregation aggregation) {
        switch (aggregation) {
            case DEFAULT_AGGREGATION:
            case SUM:
                return new SumLogHistogramCollector();
            case LAST:
                return new LastLogHistogramCollector();
            default:
                throw new UnsupportedOperationException("Unsupported log histogram aggregation: " + aggregation);
        }
    }

    private static class SumLogHistogramCollector implements PointValueCollector {
        private long count;
        private MutableLogHistogram state = new MutableLogHistogram();

        @Override
        public void reset() {
            count = 0;
            state.reset();
        }

        @Override
        public void append(AggrPoint point) {
            LogHistogram histogram = point.logHistogram;
            if (histogram == LogHistogramColumn.DEFAULT_VALUE) {
                return;
            }

            state.addHistogram(histogram);
            count += Math.max(1, point.count);
        }

        @Override
        public boolean compute(AggrPoint point) {
            point.setCount(count);
            if (count == 0) {
                LogHistogramColumn.recycle(point.logHistogram);
                point.setLogHistogram(LogHistogramColumn.DEFAULT_VALUE);
                return false;
            }
            point.setLogHistogram(state.build(LogHistogram.orNew(point.logHistogram)));
            return count != 0;
        }
    }

    private static class LastLogHistogramCollector implements PointValueCollector {
        private LogHistogram state = LogHistogram.newInstance();
        private long count = 0;

        @Override
        public void reset() {
            state.reset();
            count = 0;
        }

        @Override
        public void append(AggrPoint point) {
            LogHistogram histogram = point.logHistogram;
            if (histogram != LogHistogramColumn.DEFAULT_VALUE) {
                count += Math.max(1, point.count);
                state.copyFrom(histogram);
            }
        }

        @Override
        public boolean compute(AggrPoint point) {
            point.setCount(count);
            if (count == 0) {
                LogHistogramColumn.recycle(point.logHistogram);
                point.setLogHistogram(LogHistogramColumn.DEFAULT_VALUE);
                return false;
            }

            point.setLogHistogram(LogHistogramColumn.copy(state, point.logHistogram));
            return true;
        }
    }
}
