package ru.yandex.solomon.model.timeseries.aggregation.collectors;

import java.util.function.LongBinaryOperator;

import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.point.AggrPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class LongPointCollectors {
    public static PointValueCollector ofCounter(Aggregation aggregation) {
        if (aggregation == Aggregation.DEFAULT_AGGREGATION) {
            return ofLong(Aggregation.LAST);
        }

        return ofLong(aggregation);
    }

    public static PointValueCollector ofRate(Aggregation aggregation) {
        if (aggregation == Aggregation.DEFAULT_AGGREGATION) {
            return ofLong(Aggregation.LAST);
        }

        return ofLong(aggregation);
    }

    public static PointValueCollector ofGauge(Aggregation aggregation) {
        if (aggregation == Aggregation.DEFAULT_AGGREGATION) {
            return ofLong(Aggregation.AVG);
        }

        return ofLong(aggregation);
    }

    private static PointValueCollector ofLong(Aggregation aggregation) {
        switch (aggregation) {
            case LAST:
                return Collector.of((left, right) -> right);
            case MAX:
                return Collector.of(Long.MIN_VALUE, Math::max);
            case MIN:
                return Collector.of(Long.MAX_VALUE, Math::min);
            case COUNT:
                return Collector.of(0, (left, right) -> right, (count, state) -> count);
            case SUM:
                return Collector.of(Long::sum);
            case AVG:
                return Collector.of(0L, Long::sum, (count, sum) -> Long.divideUnsigned(sum, count));
            default:
                throw new UnsupportedOperationException("Unsupported long aggregation: " + aggregation);
        }
    }

    private static class Collector implements PointValueCollector {
        private long points = 0;
        private long aggregatedCount = 0;
        private final long init;
        private final LongBinaryOperator combiner;
        private final Finisher finisher;
        private long state;

        private Collector(long init, LongBinaryOperator combiner, Finisher finisher) {
            this.init = init;
            this.combiner = combiner;
            this.finisher = finisher;
            this.reset();
        }

        static PointValueCollector of(LongBinaryOperator combiner) {
            return of(0L, combiner);
        }

        static PointValueCollector of(long init, LongBinaryOperator combiner) {
            return of(init, combiner, (ignore, result) -> result);
        }

        static PointValueCollector of(long init, LongBinaryOperator combiner, Finisher finisher) {
            return new Collector(init, combiner, finisher);
        }

        @Override
        public void reset() {
            this.points = 0;
            this.aggregatedCount = 0;
            this.state = init;
        }

        @Override
        public void append(AggrPoint point) {
            long value = point.longValue;
            aggregatedCount += Math.max(1, point.count);
            points++;
            state = combiner.applyAsLong(state, value);
        }

        @Override
        public boolean compute(AggrPoint point) {
            point.setCount(aggregatedCount);
            if (points == 0) {
                point.setLongValue(0);
                return false;
            }

            point.setLongValue(finisher.apply(points, state));
            return true;
        }
    }

    private interface Finisher {
        long apply(long count, long state);
    }
}
