package ru.yandex.solomon.model.timeseries.aggregation.collectors;

import ru.yandex.monlib.metrics.summary.SummaryDoubleSnapshot;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.SummaryDoubleColumn;
import ru.yandex.solomon.model.type.MutableSummaryDouble;
import ru.yandex.solomon.model.type.SummaryDouble;

/**
 * @author Vladimir Gordiychuk
 */
public class SummaryDoublePointCollectors {
    public static PointValueCollector ofSummaryDouble(Aggregation aggregation) {
        switch (aggregation) {
            case DEFAULT_AGGREGATION:
            case SUM:
                return new SumSummaryDoubleCollector();
            case LAST:
                return new LastSummaryDoubleCollector();
            default:
                throw new UnsupportedOperationException("Unsupported summary aggregation: " + aggregation);
        }
    }

    private static class SumSummaryDoubleCollector implements PointValueCollector {
        private long count = 0;
        private MutableSummaryDouble state = new MutableSummaryDouble();

        @Override
        public void reset() {
            count = 0;
            state.reset();
        }

        @Override
        public void append(AggrPoint point) {
            SummaryDoubleSnapshot snapshot = point.summaryDouble;
            if (snapshot != SummaryDoubleColumn.DEFAULT_VALUE) {
                count += Math.max(1, point.count);
                state.addSummary(snapshot);
            }
        }

        @Override
        public boolean compute(AggrPoint point) {
            point.setCount(count);
            if (count == 0) {
                SummaryDouble.recycle(point.summaryDouble);
                point.setSummaryDouble(SummaryDoubleColumn.DEFAULT_VALUE);
                return false;
            }

            point.setSummaryDouble(state.snapshot(point.summaryDouble));
            return true;
        }
    }

    private static class LastSummaryDoubleCollector implements PointValueCollector {
        private SummaryDouble state = SummaryDouble.newInstance();
        private long count;

        @Override
        public void reset() {
            state.reset();
            count = 0;
        }

        @Override
        public void append(AggrPoint point) {
            if (point.summaryDouble != SummaryDoubleColumn.DEFAULT_VALUE) {
                count += Math.max(1, point.count);
                state.copyFrom(point.summaryDouble);
            }
        }

        @Override
        public boolean compute(AggrPoint point) {
            point.setCount(count);
            if (count == 0) {
                SummaryDoubleColumn.recycle(point.summaryDouble);
                point.setSummaryDouble(SummaryDoubleColumn.DEFAULT_VALUE);
                return false;
            }

            point.setSummaryDouble(SummaryDoubleColumn.copy(state, point.summaryDouble));
            return true;
        }
    }
}
