package ru.yandex.solomon.model.timeseries.aggregation.collectors;

import ru.yandex.monlib.metrics.summary.SummaryInt64Snapshot;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.SummaryInt64Column;
import ru.yandex.solomon.model.type.MutableSummaryInt64;
import ru.yandex.solomon.model.type.SummaryInt64;

/**
 * @author Vladimir Gordiychuk
 */
public class SummaryInt64PointCollectors {
    public static PointValueCollector ofSummaryInt64(Aggregation aggregation) {
        switch (aggregation) {
            case DEFAULT_AGGREGATION:
            case SUM:
                return new SumSummaryInt64Collector();
            case LAST:
                return new LastSummaryInt64Collector();
            default:
                throw new UnsupportedOperationException("Unsupported summary aggregation: " + aggregation);
        }
    }

    private static class SumSummaryInt64Collector implements PointValueCollector {
        private long count;
        private MutableSummaryInt64 state = new MutableSummaryInt64();

        @Override
        public void reset() {
            count = 0;
            state.reset();
        }

        @Override
        public void append(AggrPoint point) {
            SummaryInt64Snapshot snapshot = point.summaryInt64;
            if (snapshot != SummaryInt64Column.DEFAULT_VALUE) {
                count += Math.max(1, point.count);
                state.addSummary(snapshot);
            }
        }

        @Override
        public boolean compute(AggrPoint point) {
            point.setCount(count);
            if (count == 0) {
                point.setSummaryInt64(SummaryInt64Column.DEFAULT_VALUE);
                return false;
            }

            point.setSummaryInt64(state.snapshot());
            return true;
        }
    }

    private static class LastSummaryInt64Collector implements PointValueCollector {
        private SummaryInt64 state = SummaryInt64.newInstance();
        private long count;

        @Override
        public void reset() {
            state.reset();
            count = 0;
        }

        @Override
        public void append(AggrPoint point) {
            if (point.summaryInt64 != SummaryInt64Column.DEFAULT_VALUE) {
                count += Math.max(1, point.count);
                state.copyFrom(point.summaryInt64);
            }
        }

        @Override
        public boolean compute(AggrPoint point) {
            if (count == 0) {
                SummaryInt64Column.recycle(point.summaryInt64);
                point.setSummaryInt64(SummaryInt64Column.DEFAULT_VALUE);
                return false;
            }

            point.setSummaryInt64(SummaryInt64Column.copy(state, point.summaryInt64));
            point.setCount(count);
            return true;
        }
    }
}
