package ru.yandex.solomon.model.timeseries.decim;

import com.google.common.math.LongMath;

import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.CountColumn;
import ru.yandex.solomon.model.point.column.MergeColumn;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.aggregation.collectors.NumDenomAverageDoublePointCollector;
import ru.yandex.solomon.model.timeseries.aggregation.collectors.PointValueCollector;
import ru.yandex.solomon.model.timeseries.aggregation.collectors.PointValueCollectors;

/**
 * @author Vladimir Gordiychuk
 */
public class DecimPointValueCollector implements PointValueCollector {
    private PointValueCollector valueCollector;
    private long stepMillis;
    private long sumCount;
    private int count;
    private boolean merge;

    private DecimPointValueCollector(PointValueCollector valueCollector) {
        this.valueCollector = valueCollector;
    }

    public static DecimPointValueCollector of(MetricType type) {
        switch (type) {
            case DGAUGE:
                // TODO: drop this hack as only rate metrics will be store into correspond column and denom will be dropped (@gordiychuk)
                return new DecimPointValueCollector(new NumDenomAverageDoublePointCollector());
            default:
                return new DecimPointValueCollector(PointValueCollectors.of(type, Aggregation.DEFAULT_AGGREGATION));
        }
    }

    @Override
    public void reset() {
        count = 0;
        merge = MergeColumn.DEFAULT_VALUE;
        sumCount = CountColumn.DEFAULT_VALUE;
        stepMillis = StepColumn.DEFAULT_VALUE;
        valueCollector.reset();
    }

    @Override
    public void append(AggrPoint point) {
        merge = false; // is it valid? but compatible with old behaviour
        stepMillis = Math.max(stepMillis, point.stepMillis);
        sumCount = LongMath.saturatedAdd(sumCount, point.count);
        valueCollector.append(point);
        count++;
    }

    @Override
    public boolean compute(AggrPoint point) {
        boolean result = valueCollector.compute(point);
        point.merge = merge;
        point.count = count > 0 ? sumCount / count : 0;
        point.stepMillis = stepMillis;
        return result;
    }

    public int getCount() {
        return count;
    }
}
