package ru.yandex.solomon.model.timeseries.decim;

import java.time.Duration;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.stockpile.api.EDecimPolicy;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class DecimPoliciesPredefined {

    // default policy for pull in Solomon
    private static final DecimPolicy POLICY_5_MIN_AFTER_7_DAYS = DecimPolicy.newBuilder()
        .addPolicy(Duration.ofDays(7), Duration.ofMinutes(5))
        .build();
    // obsolete default policy for push in Solomon
    private static final DecimPolicy POLICY_KEEP_FOREVER = DecimPolicy.EMPTY;
    // default policy for Graphite
    private static final DecimPolicy POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS = DecimPolicy.newBuilder()
        .addPolicy(Duration.ofDays(30), Duration.ofMinutes(1))
        .addPolicy(Duration.ofDays(90), Duration.ofMinutes(5))
        .build();

    // policies for Graphite
    private static final DecimPolicy POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_2_MONTHS = DecimPolicy.newBuilder()
        .addPolicy(Duration.ofDays(30), Duration.ofMinutes(1))
        .addPolicy(Duration.ofDays(60), Duration.ofMinutes(5))
        .build();

    private static final DecimPolicy POLICY_5_MIN_AFTER_2_MONTHS = DecimPolicy.newBuilder()
            .addPolicy(Duration.ofDays(60), Duration.ofMinutes(5))
            .build();

    private static final DecimPolicy POLICY_5_MIN_AFTER_8_DAYS = DecimPolicy.newBuilder()
        .addPolicy(Duration.ofDays(8), Duration.ofMinutes(5))
        .build();

    private static final EDecimPolicy[] decimPolicyProtos = Arrays.stream(EDecimPolicy.values())
            .filter(decimPolicy -> decimPolicy != EDecimPolicy.UNRECOGNIZED)
            .toArray(EDecimPolicy[]::new);

    private static final EnumMap<EDecimPolicy, DecimPolicy> policiesMap = new EnumMap<>(Map.of(
        EDecimPolicy.UNDEFINED, POLICY_KEEP_FOREVER,
        EDecimPolicy.POLICY_KEEP_FOREVER, POLICY_KEEP_FOREVER,
        EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS, POLICY_5_MIN_AFTER_7_DAYS,
        EDecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS, POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS,
        EDecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_2_MONTHS, POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_2_MONTHS,
        EDecimPolicy.POLICY_5_MIN_AFTER_2_MONTHS, POLICY_5_MIN_AFTER_2_MONTHS,
        EDecimPolicy.POLICY_5_MIN_AFTER_8_DAYS, POLICY_5_MIN_AFTER_8_DAYS
    ));

    static {
        for (int i = 0; i < decimPolicyProtos.length; i++) {
            EDecimPolicy decimPolicy = decimPolicyProtos[i];
            assert decimPolicy.getNumber() == i;
            assert policiesMap.containsKey(decimPolicy);
        }
    }

    @Nonnull
    private static EDecimPolicy policyIdToProto(int id) {
        if (!isKnownId(id)) {
            throw new IllegalArgumentException("unknown policy id: " + id);
        }
        return decimPolicyProtos[id];
    }

    public static boolean isKnownId(int id) {
        return id >= 0 && id < decimPolicyProtos.length;
    }

    public static int policiesCount() {
        return decimPolicyProtos.length;
    }

    @Nonnull
    public static DecimPolicy policyFromProto(EDecimPolicy decimPolicyId) {
        return policiesMap.get(decimPolicyId);
    }

    @Nonnull
    public static DecimPolicy policyByNumber(int decimPolicyId) {
        return policyFromProto(policyIdToProto(decimPolicyId));
    }
}
